use std::{
    error::Error,
    fmt::{self, Display, Formatter},
};

use crate::{CryptoError, Entry, EntryPath, HasBuilder, States, Storer};
use async_trait::async_trait;
use futures::StreamExt;
use mongodb::{
    bson::{self, Document},
    options::ClientOptions,
    options::{FindOneOptions, FindOptions},
    Client, Database,
};
use serde::{Deserialize, Serialize};

#[derive(Debug)]
pub enum MongoStorerError {
    /// Represents an error which occurred in some internal system
    InternalError {
        source: Box<dyn Error + Send + Sync>,
    },

    /// Requested document was not found
    NotFound,
}

impl Error for MongoStorerError {
    fn source(&self) -> Option<&(dyn Error + 'static)> {
        match *self {
            MongoStorerError::InternalError { ref source } => Some(source.as_ref()),
            MongoStorerError::NotFound => None,
        }
    }
}

impl Display for MongoStorerError {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        match *self {
            MongoStorerError::InternalError { .. } => {
                write!(f, "Internal error occurred")
            }
            MongoStorerError::NotFound => {
                write!(f, "Requested document not found")
            }
        }
    }
}

impl From<MongoStorerError> for CryptoError {
    fn from(mse: MongoStorerError) -> Self {
        match mse {
            MongoStorerError::InternalError { .. } => CryptoError::InternalError {
                source: Box::new(mse),
            },
            MongoStorerError::NotFound => CryptoError::NotFound {
                source: Box::new(mse),
            },
        }
    }
}

/// Stores an instance of a mongodb-backed key storer
#[derive(Clone)]
pub struct MongoStorer {
    db_info: MongoDbInfo,
    client: Client,
    db: Database,
}

#[derive(Serialize, Deserialize, Clone)]
pub struct MongoDbInfo {
    url: String,
    db_name: String,
}

impl MongoStorer {
    /// Instantiates a mongo-backed key storer using a URL to the mongo cluster and the
    /// name of the DB to connect to.
    pub async fn new(url: &str, db_name: &str) -> Self {
        let db_info = MongoDbInfo {
            url: url.to_owned(),
            db_name: db_name.to_owned(),
        };
        let db_client_options = ClientOptions::parse_with_resolver_config(
            url,
            mongodb::options::ResolverConfig::cloudflare(),
        )
        .await
        .unwrap();
        let client = Client::with_options(db_client_options).unwrap();
        let db = client.database(db_name);
        MongoStorer {
            db_info,
            client,
            db,
        }
    }
}

#[async_trait]
impl Storer for MongoStorer {
    async fn get_indexed<T: HasBuilder>(
        &self,
        path: &str,
        index: &Option<Document>,
    ) -> Result<Entry, CryptoError> {
        let mut filter = bson::doc! { "path": path };
        if let Some(i) = index {
            filter.insert("value", i);
        }

        let filter_options = FindOneOptions::builder().build();

        match self
            .db
            .collection_with_type::<Entry>("entries")
            .find_one(filter, filter_options)
            .await
        {
            Ok(Some(entry)) => Ok(entry),
            Ok(None) => Err(MongoStorerError::NotFound.into()),
            Err(e) => Err(MongoStorerError::InternalError {
                source: Box::new(e),
            }
            .into()),
        }
    }

    async fn list_indexed<T: HasBuilder + Send>(
        &self,
        path: &str,
        skip: i64,
        page_size: i64,
        index: &Option<Document>,
    ) -> Result<Vec<Entry>, CryptoError> {
        let mut filter = bson::doc! { "path": path };
        if let Some(i) = index {
            filter.insert("value", i);
        }
        let filter_options = FindOptions::builder().skip(skip).limit(page_size).build();

        match self
            .db
            .collection_with_type::<Entry>("entries")
            .find(filter, filter_options)
            .await
        {
            Ok(cursor) => Ok(cursor
                .filter_map(|result| async move {
                    match result {
                        Ok(entry) => Some(entry),
                        Err(_) => None,
                    }
                })
                .collect::<Vec<Entry>>()
                .await),
            Err(e) => Err(MongoStorerError::InternalError {
                source: Box::new(e),
            }
            .into()),
        }
    }

    async fn create(&self, path: EntryPath, value: States) -> Result<bool, CryptoError> {
        let filter = bson::doc! { "path": &path };
        let entry = Entry { path, value };
        let filter_options = mongodb::options::ReplaceOptions::builder()
            .upsert(true)
            .build();

        match self
            .db
            .collection_with_type::<Entry>("entries")
            .replace_one(filter, entry, filter_options)
            .await
        {
            Ok(_) => Ok(true),
            Err(e) => Err(MongoStorerError::InternalError {
                source: Box::new(e),
            }
            .into()),
        }
    }
}
