use std::{
    error::Error,
    fmt::{self, Display, Formatter},
};

use crate::{CryptoError, Entry, EntryPath, HasBuilder, States, Storer};
use async_trait::async_trait;
use mongodb::bson::Document;

#[derive(Debug)]
pub enum RedactStorerError {
    /// Represents an error which occurred in some internal system
    InternalError {
        source: Box<dyn Error + Send + Sync>,
    },

    /// Requested document was not found
    NotFound,
}

impl Error for RedactStorerError {
    fn source(&self) -> Option<&(dyn Error + 'static)> {
        match *self {
            RedactStorerError::InternalError { ref source } => Some(source.as_ref()),
            RedactStorerError::NotFound => None,
        }
    }
}

impl Display for RedactStorerError {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        match *self {
            RedactStorerError::InternalError { .. } => {
                write!(f, "Internal error occurred")
            }
            RedactStorerError::NotFound => {
                write!(f, "Requested document not found")
            }
        }
    }
}

impl From<RedactStorerError> for CryptoError {
    fn from(rse: RedactStorerError) -> Self {
        match rse {
            RedactStorerError::InternalError { .. } => CryptoError::InternalError {
                source: Box::new(rse),
            },
            RedactStorerError::NotFound => CryptoError::NotFound {
                source: Box::new(rse),
            },
        }
    }
}

#[derive(Clone)]
pub struct RedactStorer {
    url: String,
}

/// Stores an instance of a redact-backed key storer.
/// The redact-store server is an example implementation of a redact storage backing.
impl RedactStorer {
    pub fn new(url: &str) -> Self {
        Self {
            url: url.to_owned(),
        }
    }
}

#[async_trait]
impl Storer for RedactStorer {
    async fn get_indexed<T: HasBuilder>(
        &self,
        path: &str,
        index: &Option<Document>,
    ) -> Result<Entry, CryptoError> {
        let mut req_url = format!("{}/{}?", &self.url, path);
        if let Some(i) = index {
            req_url.push_str(format!("index={}", i).as_ref());
        }
        match reqwest::get(&req_url).await {
            Ok(r) => Ok(r
                .error_for_status()
                .map_err(|source| -> CryptoError {
                    if source.status() == Some(reqwest::StatusCode::NOT_FOUND) {
                        RedactStorerError::NotFound.into()
                    } else {
                        RedactStorerError::InternalError {
                            source: Box::new(source),
                        }
                        .into()
                    }
                })?
                .json::<Entry>()
                .await
                .map_err(|source| -> CryptoError {
                    RedactStorerError::InternalError {
                        source: Box::new(source),
                    }
                    .into()
                })?),
            Err(source) => Err(RedactStorerError::InternalError {
                source: Box::new(source),
            }
            .into()),
        }
    }

    async fn list_indexed<T: HasBuilder + Send>(
        &self,
        path: &str,
        skip: i64,
        page_size: i64,
        index: &Option<Document>,
    ) -> Result<Vec<Entry>, CryptoError> {
        let mut req_url = format!(
            "{}/{}?skip={}&page_size={}",
            &self.url, path, skip, page_size
        );
        if let Some(i) = index {
            req_url.push_str(format!("&index={}", i).as_ref());
        }
        match reqwest::get(&req_url).await {
            Ok(r) => Ok(r
                .error_for_status()
                .map_err(|source| -> CryptoError {
                    if source.status() == Some(reqwest::StatusCode::NOT_FOUND) {
                        RedactStorerError::NotFound.into()
                    } else {
                        RedactStorerError::InternalError {
                            source: Box::new(source),
                        }
                        .into()
                    }
                })?
                .json::<Vec<Entry>>()
                .await
                .map_err(|source| -> CryptoError {
                    RedactStorerError::InternalError {
                        source: Box::new(source),
                    }
                    .into()
                })?),
            Err(source) => Err(RedactStorerError::InternalError {
                source: Box::new(source),
            }
            .into()),
        }
    }

    async fn create(&self, path: EntryPath, value: States) -> Result<bool, CryptoError> {
        let entry = Entry { path, value };
        let client = reqwest::Client::new();
        match client
            .post(&format!("{}/", self.url))
            .json(&entry)
            .send()
            .await
        {
            Ok(_) => Ok(true),
            Err(source) => Err(RedactStorerError::InternalError {
                source: Box::new(source),
            }
            .into()),
        }
    }
}
