#![allow(unused)]

use recordbox::traits::{ Record, RecordId, UniqueRecordId, FastUniqueRecordId };
use std::{
    array::TryFromSliceError, ops::Deref,
    convert::{ TryFrom, Infallible }
};


/// A key
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Key {
    /// The bytes
    inner: Vec<u8>
}
impl Key {
    /// Creates a key
    pub fn new<T>(bytes: T) -> Self where T: AsRef<[u8]> {
        Self { inner: bytes.as_ref().into() }
    }
    /// Creates a key from a hex literal
    pub fn from_hex<T>(hex: T) -> Self where T: AsRef<[u8]> {
        let inner = hex::decode(hex).expect("Invalid hex literal");
        Self { inner }
    }

    /// The bytes as slice
    pub fn as_slice(&self) -> &[u8] {
        &self.inner
    }
}
impl From<Key> for Vec<u8> {
    fn from(record: Key) -> Self {
        record.inner
    }
}


/// A raw byte record
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct ByteRecord {
    /// The bytes
    inner: Vec<u8>
}
impl ByteRecord {
    /// Creates a new byte record
    pub fn new<T>(bytes: T) -> Self where T: AsRef<[u8]> {
        Self { inner: bytes.as_ref().into() }
    }
    /// Creates a byte record from a hex literal
    pub fn from_hex<T>(hex: T) -> Self where T: AsRef<[u8]> {
        let inner = hex::decode(hex).expect("Invalid hex literal");
        Self { inner }
    }

    /// The bytes as slice
    pub fn as_slice(&self) -> &[u8] {
        &self.inner
    }
}
impl Record for ByteRecord {
    type Error = Infallible;

    fn encode(&self) -> std::result::Result<Vec<u8>, Self::Error> {
        Ok(self.inner.clone())
    }
    fn decode<D>(encoded: D) -> std::result::Result<Self, Self::Error> where D: AsRef<[u8]> {
        Ok(Self::new(encoded.as_ref()))
    }
}


/// A raw byte record ID
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct ByteRecordId {
    /// The bytes
    inner: Vec<u8>
}
impl ByteRecordId {
    /// Creates a new byte record ID
    pub fn new<T>(bytes: T) -> Self where T: AsRef<[u8]> {
        Self { inner: bytes.as_ref().into() }
    }
    /// Creates a byte record ID from a hex literal
    pub fn from_hex<T>(hex: T) -> Self where T: AsRef<[u8]> {
        let inner = hex::decode(hex).expect("Invalid hex literal");
        Self { inner }
    }

    /// The bytes as slice
    pub fn as_slice(&self) -> &[u8] {
        &self.inner
    }
}
impl RecordId for ByteRecordId {
    type Error = Infallible;

    fn encode(&self) -> std::result::Result<Vec<u8>, Self::Error> {
        Ok(self.inner.clone())
    }
}
impl UniqueRecordId for ByteRecordId {
    /* No members to implement */
}
impl FastUniqueRecordId for ByteRecordId {
    type Error = TryFromSliceError;

    fn encode_fast<const LEN: usize>(&self) -> std::result::Result<[u8; LEN], <Self as FastUniqueRecordId>::Error> {
        <[u8; LEN]>::try_from(self.inner.as_slice())
    }
}


/// A ciphertext
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Ciphertext {
    /// The bytes
    inner: Vec<u8>
}
impl Ciphertext {
    /// Creates a ciphertext
    pub fn new<T>(bytes: T) -> Self where T: AsRef<[u8]> {
        Self { inner: bytes.as_ref().into() }
    }
    /// Creates a ciphertext from a hex literal
    pub fn from_hex<T>(hex: T) -> Self where T: AsRef<[u8]> {
        let inner = hex::decode(hex).expect("Invalid hex literal");
        Self { inner }
    }

    /// The bytes as slice
    pub fn as_slice(&self) -> &[u8] {
        &self.inner
    }
}