//! Implements a ChaChaPoly-IETF-based record box

#![cfg(feature = "impl_chachapoly")]

use crate::{
    error::Result,
    traits::{ Record, FastUniqueRecordId, FastRecordbox }
};
use crypto_api_chachapoly::{ ChachaPolyIetf, crypto_api::cipher::AeadCipher };


/// A ChaChaPoly-IETF-based record box
pub struct RecordboxChachaPoly {
    /// The ChaChaPoly implementation
    aead: Box<dyn AeadCipher>,
    /// The key to use
    key: Vec<u8>
}
impl RecordboxChachaPoly {
    /// The name of the implementation
    pub const NAME: &'static str = "de.KizzyCode.RecordBox.v1.ChaChaPolyIETF.BD62112A-ADF5-4171-ABC6-EFE559F06B04";
    /// The nonce length required by the implementation
    pub const NONCE_LEN: usize = 12;

    /// Creates a new ChaChaPoly-IETF record box
    pub fn new<K>(key: K) -> Result<Self> where K: Into<Vec<u8>> {
        // Create the AEAD instance and ingest the key
        let aead = ChachaPolyIetf::aead_cipher();
        let key = key.into();
        
        // Validate the key and create self
        if !aead.info().key_len_r.contains(&key.len()) {
            Err(einval!("Invalid key length for ChaChaPoly: {}", key.len()))?;
        }
        Ok(Self { aead, key })
    }
}
impl FastRecordbox for RecordboxChachaPoly {
    fn name(&self) -> &'static str {
        Self::NAME
    }
    fn nonce_len(&self) -> usize {
        Self::NONCE_LEN
    }

    fn seal<R, I>(&self, record: R, id: I) -> Result<Vec<u8>> where R: Record, I: FastUniqueRecordId {
        // Encode record and derive nonce
        let plaintext = record.encode()
            .map_err(|e| ecoding!("Failed to encode record ({})", e))?;
        let nonce: [u8; Self::NONCE_LEN] = id.encode_fast()
            .map_err(|e| ecoding!("Failed to derive nonce ({})", e))?;
        
        // Create a buffer and seal the ciphertext
        let mut ciphertext_full = vec![0; self.aead.encrypted_len_max(plaintext.len())];
        let ciphertext_len = self.aead.seal_to(&mut ciphertext_full, &plaintext, b"", &self.key, &nonce)
            .map_err(|e| ecrypto!("Failed to seal record ({})", e))?;
        Ok(ciphertext_full[..ciphertext_len].to_vec())
    }
    fn open<C, I, R>(&self, ciphertext: C, id: I) -> Result<R> where C: AsRef<[u8]>, I: FastUniqueRecordId, R: Record {
        // Derive the nonce and open the ciphertext
        let nonce: [u8; Self::NONCE_LEN] = id.encode_fast()
            .map_err(|e| ecoding!("Failed to derive nonce ({})", e))?;
        let mut plaintext_full = vec![0; ciphertext.as_ref().len()];
        let plaintext_len = self.aead.open_to(&mut plaintext_full, ciphertext.as_ref(), b"", &self.key, &nonce)
            .map_err(|e| ecrypto!("Failed to open record ({})", e))?;
        
        // Decode the record
        let record: R = R::decode(&plaintext_full[..plaintext_len])
            .map_err(|e| ecoding!("Failed to decode record ({})", e))?;
        Ok(record)
    }
}
