//! Implements the crate's error types

use ebacktrace::define_error;
use std::{
    error, result,
    fmt::{ self, Display, Formatter }
};


/// Creates a new variant
#[macro_export] macro_rules! e {
    ($kind:expr, $($arg:tt)*) => ({ $crate::error::ErrorImpl::with_string($kind, format!($($arg)*)) })
}
/// Creates a new `ErrorImpl::InvalidValue` kind
#[macro_export] macro_rules! einval {
    ($($arg:tt)*) => ({ e!($crate::error::ErrorKind::InvalidValue, $($arg)*) });
}
/// Creates a new `ErrorImpl::CryptoError` kind
#[macro_export] macro_rules! ecrypto {
    ($($arg:tt)*) => ({ e!($crate::error::ErrorKind::CryptoError, $($arg)*) });
}
/// Creates a new `ErrorImpl::CodingError` kind
#[macro_export] macro_rules! ecoding {
    ($($arg:tt)*) => ({ e!($crate::error::ErrorKind::CodingError, $($arg)*) });
}


/// The error kind
#[derive(Debug)]
pub enum ErrorKind {
    /// A value or argument is invalid
    InvalidValue,
    /// A cryptographic error occurred
    CryptoError,
    /// An en-/decoding error
    CodingError
}
impl Display for ErrorKind {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        match self {
            Self::InvalidValue => write!(f, "A value or argument is invalid"),
            Self::CryptoError => write!(f, "A cryptographic error occurred"),
            Self::CodingError => write!(f, "An en-/decoding error")
        }
    }
}
impl error::Error for ErrorKind {
    /* Nothing to implement */
}


// Define our custom error type
define_error!(ErrorImpl);


/// A nice typealias for our custom error
pub type Error = ErrorImpl<ErrorKind>;
/// A nice typealias for a `Result` with our custom error
pub type Result<T = ()> = result::Result<T, ErrorImpl<ErrorKind>>;
