//! This crate provides a bindgen mapping to the low-level C-API of librealsense2.
//!
//! In that respect, it is fairly straightforward in how realsense-sys maps types from the C-API to
//! Rust, as nothing particularly unique is done other than running `bindgen` to generate the
//! bindings.
//!
//! # Getting started
//!
//! This library is probably not terribly useful on its own, and is meant to be paired with the
//! high-level realsense-rust. However, if you're just looking for your own low-level unsafe
//! wrapper, then realsense-sys is what you want! You can get started by including the following in
//! your `Cargo.toml`:
//!
//! ```text
//! [dependencies]
//! realsense-rust = "0.5.0"
//! ```
//!
//! followed by this `use` statement in your code:
//!
//! ```no_run
//! use realsense_sys as sys;
//! ```
//!
//! The `sys` alias is used extensively throughout the realsense-rust crate, so you'll often see
//! code of the form `sys::rs2_XXX`.
//!
//! # Understanding lifetimes
//!
//! This library is generated by using `bindgen` on a set of C headers, usually located at
//! `/usr/include/librealsense2/rs.h` (or wherever you installed librealsense2). Inherently, this
//! makes most of the code in this module unsafe, since it is relying on the underlying C library
//! to define what the lifetimes are for every data type.
//!
//! librealsense2 does not always do the best job at documenting what the lifetime of an object is.
//! Understand that the library is primarily a C++ library, with a C-compatible API built on top of
//! it. This means that while some guarantees about lifetimes can be made by some types, these
//! guarantees are not always explicit. By that, I mean that many quantities in the C++ library are
//! managed via C++ `shared_ptr` or `unique_ptr`. However, the C API on top of this cannot expose
//! these types, and it is often unclear if a pointer you get in the C API is a result of calling
//! `shared_ptr::get()` or `unique_ptr::get()` under the hood. A good example of this is
//! `rs2_get_frame_sensor`, which will give you a pointer to a sensor type from a `shared_ptr`
//! under the hood. As a result, you do not need to manually manage this pointer and can just drop
//! it whenever as the `shared_ptr` under the hood will delete the resource when it is no longer
//! held. However, if you get the sensor from a sensor list in the low-level API by calling
//! `rs2_create_sensor` then you will notice that this pointer is allocated with `new`, and if you
//! were using this then it needs to be deleted by a call to `rs2_delete_sensor`. In both cases you
//! get a `*mut rs2_sensor` from the wrapper, but the lifetime and ownership information is dealt
//! with very differently. This makes the API fairly difficult to navigate.
//!
//! - [`rs2_get_frame_sensor`](https://github.com/IntelRealSense/librealsense/blob/master/src/rs.cpp#L922)
//! - [`rs2_create_sensor`](https://github.com/IntelRealSense/librealsense/blob/master/src/rs.cpp#L308)
//!
//! In general, reading through `bindings.rs` in the repo will be useful in describing the
//! documentation that Intel provides around every function in the C-API. However, you may find
//! that such documentation is insufficient to understand the lifetimes since not every function
//! documents the proper ownership. As a result you end up needing to understand librealsense2 in
//! C, C++, and Rust in order to utilize the realsense-sys library safely and effectively. If you
//! do find yourself looking for an entry point into the librealsense2 C-API, we highly suggest
//! starting at [this file](https://github.com/IntelRealSense/librealsense/blob/master/src/rs.cpp)
//! and working your way out via each type.
//!
//! If this seems like a lot of effor to you (it truly is!), we highly suggest using the
//! realsense-rust wrapper, which attempts to abstract over these and provide a high-level,
//! Rust-native API that avoids unsafe code.
//!
//! # Regenerating bindings
//!
//! The bindings should be uniquely generated for every new release of librealsense2. Currently,
//! the bindings in this crate are compatible with librealsense2 version 2.41.0.
//!
//! If you want to regenerate the bindings for your version of realsense, enable the
//! buildtime-bindgen feature of the crate. Note that in general this can break things if the API
//! for your version of librealsense2 is different with regards to what the realsense-rust crate is
//! expecting.

// Allow all warnings here -- Bindgen generates this file, we really don't care about individual
// warnings since we can't really do much about them, we'd have to fix bindgen upstream or
// librealsense2 itself.
#![allow(warnings)]
#![allow(clippy::missing_docs_in_private_items)]

include!("../bindings/bindings.rs");
