// Copyright 2022 tison <wander4096@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

use std::sync::{Arc, Mutex};

use crate::{subscriber::Subscriber, subscription::Subscription};

pub struct ScalarSubscription<'s, T, S: Subscriber<Item = T>> {
    state: Arc<Mutex<State<T>>>,
    subscriber: &'s S,
}

enum State<T> {
    NoRequest(T),
    Requested,
    Cancelled,
}

impl<T> State<T> {
    fn into_inner(self) -> Option<T> {
        match self {
            State::NoRequest(t) => Some(t),
            _ => None,
        }
    }
}

impl<'s, T, S> ScalarSubscription<'s, T, S>
where
    S: Subscriber<Item = T>,
{
    pub fn new(t: T, subscriber: &'s S) -> Self {
        ScalarSubscription {
            state: Arc::new(Mutex::new(State::NoRequest(t))),
            subscriber,
        }
    }
}

impl<'s, T, S> Subscription for ScalarSubscription<'s, T, S>
where
    S: Subscriber<Item = T>,
{
    fn request(&self, _: u64) {
        if let Ok(mut state) = self.state.lock() {
            if let State::NoRequest(_) = *state {
                let t = std::mem::replace(&mut *state, State::Requested);
                self.subscriber.on_next(t.into_inner().unwrap());
                self.subscriber.on_completed();
            }
        }
    }

    fn cancel(&self) {
        if let Ok(mut state) = self.state.lock() {
            *state = State::Cancelled;
        }
    }
}
