// Copyright 2022 tison <wander4096@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

use std::sync::{
    atomic::{AtomicBool, AtomicU64, Ordering},
    Arc, Mutex,
};

use crate::{subscriber::Subscriber, subscription::Subscription};

pub struct IterSubscription<'s, T, I, S>
where
    I: Iterator<Item = T>,
    S: Subscriber<Item = T>,
{
    iter: Arc<Mutex<I>>,
    requested: AtomicU64,
    cancel: AtomicBool,
    subscriber: &'s S,
}

impl<'s, T, I, S> IterSubscription<'s, T, I, S>
where
    I: Iterator<Item = T>,
    S: Subscriber<Item = T>,
{
    pub fn new(iter: I, subscriber: &'s S) -> Self {
        IterSubscription {
            iter: Arc::new(Mutex::new(iter)),
            requested: AtomicU64::new(0),
            cancel: AtomicBool::new(false),
            subscriber,
        }
    }
}

impl<'s, T, I, S> Subscription for IterSubscription<'s, T, I, S>
where
    I: Iterator<Item = T>,
    S: Subscriber<Item = T>,
{
    fn request(&self, n: u64) {
        let mut e = 0;
        let mut r = n;
        let subscriber = self.subscriber;
        let mut iter = match self.iter.lock() {
            Ok(i) => i,
            Err(_) => return,
        };

        loop {
            let r = self.requested.load(Ordering::SeqCst);
            if r == u64::MAX {
                break;
            }

            let nr = self.requested.compare_exchange(
                r,
                r.saturating_add(n),
                Ordering::SeqCst,
                Ordering::SeqCst,
            );

            if nr.is_ok() {
                break;
            }
        }

        loop {
            while e != r {
                if self.cancel.load(Ordering::SeqCst) {
                    return;
                }

                match iter.next() {
                    None => {
                        if !self.cancel.load(Ordering::SeqCst) {
                            subscriber.on_completed();
                        }
                        return;
                    }
                    Some(t) => {
                        if self.cancel.load(Ordering::SeqCst) {
                            return;
                        }

                        subscriber.on_next(t);
                    }
                }

                e += 1;
            }

            r = self.requested.load(Ordering::SeqCst);
            if e == r {
                r = self.requested.fetch_sub(e, Ordering::SeqCst);
                r -= e;
                if r == 0 {
                    return;
                }
                e = 0;
            }
        }
    }

    fn cancel(&self) {
        self.cancel.store(true, Ordering::SeqCst);
    }
}
