// Copyright 2022 tison <wander4096@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

use crate::{publisher::Publisher, subscriber::Subscriber, subscription::IterSubscription};

pub fn from_iter<T, I: Iterator<Item = T>>(iter: I) -> FlowableFromIter<T, I> {
    FlowableFromIter { iter }
}

pub struct FlowableFromIter<T, I: Iterator<Item = T>> {
    iter: I,
}

impl<T, I: Iterator<Item = T>> Publisher for FlowableFromIter<T, I> {
    type Item = T;

    fn subscribe<E, S: Subscriber<Item = Self::Item, Error = E>>(self, subscriber: S) {
        subscriber.on_subscribe(&IterSubscription::new(self.iter, &subscriber));
    }
}

#[cfg(test)]
mod tests {
    use crate::{flowable::iter::from_iter, publisher::Publisher, subscriber::MockSubscriber};

    #[test]
    fn it_works() {
        for i in vec![0, 1, 2, 3, 4, u64::MAX].into_iter() {
            let flowable = from_iter(vec![1, 2, 3].into_iter());
            flowable.subscribe(MockSubscriber::new(i));
        }
    }
}
