use sha2::Sha256;
use clap::ArgMatches;
use git2::Repository;
use log::{debug, info, warn};
use hmac::{Hmac, Mac, NewMac};
use crate::config::DeployConfig;
use serde::{Deserialize, Serialize};
use tide::{Error, Request, Response};

#[derive(Debug, Clone)]
struct RunData {
    git_ref: String,
    exec: Option<String>,
    commands: Vec<String>,
    verify: bool,
    key: String,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct PushEvent {
    #[serde(rename = "ref")]
    pub r#ref: String,
}

pub async fn run_server(m: &'_ ArgMatches<'_>) -> Result<(), &'static str> {
    let ref_name = find_current_head()?;
    let cfg = get_config(m)?;

    let bind_address = format!(
        "{}:{}",
        m.value_of("ip").unwrap_or(&cfg.listen_address),
        m.value_of("port").unwrap_or(&cfg.listen_port.to_string())
    );

    let run_data = RunData {
        git_ref: ref_name,
        exec: m.value_of("exec").map(|s| s.to_string()),
        verify: !m.is_present("no-verify"),
        key: get_security_key(m, &cfg)?,
        commands: cfg.on_push.unwrap_or_default(),
    };

    let mut app = tide::with_state(run_data);
    app.at("/push").post(on_push);

    if m.is_present("no-verify") {
        warn!("You are using `no-verify` flag. This may significantly lower security of your deployment.");
    }

    info!("Beginning to listen at: `{}`", &bind_address);
    let _ = app.listen(bind_address).await;

    Ok(())
}

async fn on_push(mut r: Request<RunData>) -> tide::Result<Response> {
    let body = parse_request_body(&mut r).await?;

    if r.state().verify {
        verify_request(&r, &body).await?;
    }

    let delivery_id = unwrap_header_from_request(&r, "X-GitHub-Delivery")?;
    let content_type: String = unwrap_header_from_request(&r, "Content-Type")?;

    let event: PushEvent;

    if content_type == "application/json" {
        event = r.body_json().await?;
    } else if content_type == "application/x-www-form-urlencoded" {
        let body = r.body_string().await?;
        if let Some((_, encoded)) = body.split_once('=') {
            if let Ok(decoded) = urlencoding::decode(encoded) {
                if let Ok(e) = serde_json::from_str(&decoded) {
                    event = e;
                } else {
                    debug!("Invalid x-www-form-urlencoded JSON body received");
                    return Err(Error::from_str(400, "Invalid urlencoded json body"));
                }
            } else {
                debug!("Invalid x-www-form-urlencoded string");
                return Err(Error::from_str(400, "Invalid urlencoded string"));
            }
        } else {
            debug!("Invalid x-www-form-urlencoded body received");
            return Err(Error::from_str(400, "Invalid body"));
        }
    } else {
        debug!("Unsupported Content-Type received: {}", content_type);
        return Err(Error::from_str(400, "Unknown Content-Type"));
    }

    process_push_event(&r, event, delivery_id)
}

async fn parse_request_body(r: &mut Request<RunData>) -> Result<Vec<u8>, Error> {
    if let Ok(body) = r.body_bytes().await {
        r.set_body(body.clone());

        Ok(body)
    } else {
        Err(Error::from_str(400, "Invalid request body"))
    }
}

fn process_push_event(r: &Request<RunData>, e: PushEvent, delivery_id: String) -> tide::Result {
    debug!(
        "Event branch: {}. Tracked branch: {}",
        e.r#ref,
        &r.state().git_ref
    );

    if e.r#ref != r.state().git_ref {
        Ok("Ignored. Diffrent branch is tracked.".into())
    } else {
        print!(
            "\nINFO: Start of delivery dispatch: `{}`\n==========",
            delivery_id
        );
        async_std::task::spawn({
            let cmds = r.state().commands.clone();

            async move {
                cmds.into_iter().for_each(|s| {
                    print!("\n$ {}:\n", &s);
                    let args = s.split_whitespace().collect::<Vec<_>>();
                    if !args.is_empty() {
                        if let Ok(mut c) = std::process::Command::new(args.first().unwrap())
                            .args(args.iter().skip(1))
                            .spawn()
                        {
                            if let Ok(status) = c.wait() {
                                if !status.success() {
                                    warn!("Process {} failed with code: {}.", &s, status);
                                }
                            }
                        } else {
                            warn!("Failed to spawn the process: {}.", &s);
                        }
                    }
                });
                println!(
                    "==========\nINFO: End of delivery dispatch: `{}`\n",
                    delivery_id
                );
            }
        });

        Ok("Dispatched".into())
    }
}

fn unwrap_header_from_request(r: &Request<RunData>, header_name: &str) -> Result<String, Error> {
    if let Some(header) = r.header(header_name) {
        Ok(header.as_str().to_string())
    } else {
        debug!("`{}` is not present in request.", header_name);
        Err(Error::from_str(
            400,
            format!("Missing `{}` header", header_name),
        ))
    }
}

// Verifies request for valid signature and User-Agent
async fn verify_request(r: &Request<RunData>, body: &[u8]) -> Result<(), Error> {
    // Github User-Agent always starts with `GitHub-Hookshot/`
    if !unwrap_header_from_request(r, "User-Agent")?.starts_with("GitHub-Hookshot/") {
        return Err(Error::from_str(403, "Malformed request"));
    }

    // Format of header is `sha256=SIGNATURE`
    let signature = unwrap_header_from_request(r, "x-hub-signature-256")?;

    // Verifies the signature of delivery
    let mut mac = Hmac::<Sha256>::new_from_slice(r.state().key.as_bytes()).unwrap();
    mac.update(body);
    if let Some((_, hex)) = signature.as_str().split_once('=') {
        if let Ok(bytes) = hex::decode(hex) {
            if mac.verify(bytes.as_ref()).is_ok() {
                Ok(())
            } else {
                debug!("Signature mismatch");
                Err(Error::from_str(403, "Signature mismatch"))
            }
        } else {
            debug!("Invalid signature type");
            Err(Error::from_str(400, "Invalid signature type"))
        }
    } else {
        debug!("Invalid signature format");
        Err(Error::from_str(400, "Invalid signature format"))
    }
}

// This function returns security-key retrived from either
// config file, input argument or enviroment variable.
// Order is: ARGUMENT - ENV_VARIABLE(if not file prefered) - CONFIG-FILE
#[allow(clippy::unnecessary_unwrap)]
fn get_security_key(m: &ArgMatches, cfg: &DeployConfig) -> Result<String, &'static str> {
    if let Some(key) = m.value_of("security-key") {
        return Ok(key.to_string());
    }

    let file_read_key = cfg.security_key.as_ref().cloned();
    let environment_key = std::env::var("RDKEY").ok();
    let prefer_file = m.is_present("prefer-file");

    if !prefer_file && environment_key.is_some() {
        Ok(environment_key.unwrap())
    } else if (prefer_file && file_read_key.is_some()) || environment_key.is_none() {
        Ok(file_read_key.unwrap())
    } else {
        Err("Failed to discover security-key.")
    }
}

// Recursivly finds repository in parent folders
fn get_config(m: &ArgMatches) -> Result<DeployConfig, &'static str> {
    if let Ok(text) = std::fs::read_to_string(m.value_of("config-path").unwrap()) {
        if let Ok(config) = serde_yaml::from_str::<DeployConfig>(&text) {
            Ok(config)
        } else {
            Err("Invalid configuration found at specified path. Try overwriting config with `rdeploy new -o`")
        }
    } else {
        Err("Config missing at specified path. Try running `rdeploy new` or `rdeploy run --config 'path'`.")
    }
}

// Each delivery will be verified with `ref` value
fn find_current_head() -> Result<String, &'static str> {
    let path = std::fs::canonicalize(".").expect("Failed to canonicalize the path.");
    let mut path = path.as_path();
    let repo: Repository;

    loop {
        if let Ok(r) = Repository::open(path) {
            repo = r;
            break;
        } else if let Some(parent) = path.parent() {
            path = parent
        } else {
            return Err("Failed to discover git repository in current folder. (Note: Went up recursive)");
        }
    }

    let s = repo
        .head()
        .expect("No head found in git repository")
        .name()
        .expect("Head without name found in git repository.")
        .to_string();

    Ok(s)
}
