use rand::Rng;
use std::io::Write;
use clap::ArgMatches;
use tide::log::error;
use std::net::IpAddr;
use log::{info, warn};
use std::path::PathBuf;
use std::fs::DirBuilder;
use crate::config::DeployConfig;

pub fn create_new_config(m: &ArgMatches) {
    if let Some(ip) = m.value_of("ip") {
        if ip.parse::<IpAddr>().is_err() {
            error!("Not a valid IP address was specified");
            return;
        }
    }

    let cfg = DeployConfig {
        // We can unwrap here because `clap` have these fields a default value assigned
        listen_address: m.value_of("ip").unwrap().into(),
        listen_port: m
            .value_of("port")
            .unwrap()
            .parse()
            .expect("Not a valid PORT value provided."),
        security_key: Some(
            m.value_of("secure-key")
                .map(|v| v.to_string())
                .unwrap_or_else(generate_new_secure_key),
        ),
        on_push: None,
    };

    let config_path = m.value_of("config-path").unwrap();
    if std::fs::metadata(config_path).is_ok() && !m.is_present("overwrite") {
        let mut response = String::new();
        warn!("Configuration file was found in current directory. Do you want to override it? (y/n): ");
        // We need to flush the buffer because of following stdio read
        let _ = std::io::stdout().flush();

        let _ = std::io::stdin().read_line(&mut response);

        if response.trim().to_lowercase() != "y" {
            println!("Aborted.");
            return;
        }
    }

    if DirBuilder::new()
        .recursive(true)
        .create(
            config_path
                .parse::<PathBuf>()
                .expect("Not a path")
                .parent()
                .expect("Cannot use root folder"),
        )
        .is_err()
    {
        error!("Failed to create configuration file.");
        return;
    }

    let ser = serde_yaml::to_string(&cfg).unwrap();
    std::fs::write(config_path, &ser)
        .expect("Couldn't write the file `deployconf.yaml`. Wrong permissions?");

    info!("Your config was created at `deployconf.yaml`");
    if !m.is_present("key") {
        info!("secure-key was randomly generated. Make sure updated the old one in your repository settings.");
    }
}

pub fn generate_new_secure_key() -> String {
    static ALPHABET: &str = "AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz0123456789";

    // Creates new secure key from random characters
    (0..56)
        .map(|_| {
            ALPHABET
                .chars()
                .nth(rand::thread_rng().gen_range(0..ALPHABET.len()))
                .unwrap()
        })
        .collect()
}
