#![warn(missing_debug_implementations)]

use clap::{App, Arg, SubCommand};
use log::{debug, error, LevelFilter};

mod new;
mod run;
mod config;

#[async_std::main]
async fn main() {
    fern::Dispatch::new()
        .filter(|m| m.target().contains("rdeploy"))
        .level(if cfg!(debug_assertions) {
            LevelFilter::Debug
        } else {
            LevelFilter::Error
        })
        .format(|out, args, r| {
            out.finish(format_args!("{}: {}", r.level(), args));
        })
        .chain(std::io::stdout())
        .apply()
        .expect("Failed to initialize logging environment!");

    let app = App::new("rdeploy")
        .name("RDeploy")
        .version("0.1.0")
        .about("\nQuick start: \n\
        1. Run `rdeploy new` to create new config.\n\
        2. In `on_push` put commands you want to execute on push event.\n\
        3. Setup your webhook in your GitHub repository.\n\
        4. When you are ready, execute `rdeploy run` to start listening for events.")
        .author("@sy1ntexx")
        .subcommands(
            [
                SubCommand::with_name("run")
                    .about("Starts listener at IP:PORT. When gets notified from github, \
                     proceeds to execute commands from `on_push` array in config.")
                    .args(
                        &[
                            Arg::with_name("port")
                                .help("Overrides port read from the config.")
                                .short("p")
                                .long("port")
                                .takes_value(true),
                            Arg::with_name("ip")
                                .help("Overrides listen address read from the config.")
                                .short("i")
                                .long("ip")
                                .takes_value(true),
                            Arg::with_name("no-verify")
                                .help("Disables verification of received events. \
                                WARNING: THIS OPTION MIGHT LOWER SECURITY OF APPLICATION")
                                .long("no-verify"),
                            Arg::with_name("security-key")
                                .help("Overrides security-key read from the config. \
                                RDeploy will also look for environment variable with name `RDKEY`.")
                                .long("key")
                                .short("k")
                                .takes_value(true),
                            // Arg::with_name("exec")
                            //     .help("Instead of executing commands of `on_push` array, \
                            //     webhook proceeds to start the process on path.")
                            //     .short("E")
                            //     .long("exec")
                            //     .takes_value(true),
                            Arg::with_name("config-path")
                                .help("Overrides default config path.")
                                .short("C")
                                .long("config")
                                .default_value("deployconf.yaml")
                                .takes_value(true),
                            Arg::with_name("prefer-file")
                                .help("Will you file security-key even if `RDKEY` environment variable is set.")
                                .long("prefer-file")
                        ]
                    ),
                SubCommand::with_name("new")
                    .about("Creates new config for repository in current folder")
                    .args(
                        &[
                            Arg::with_name("port")
                                .help("Port to listen on.")
                                .short("p")
                                .long("port")
                                .default_value("4200")
                                .takes_value(true),
                            Arg::with_name("ip")
                                .help("Address to bind to.")
                                .short("i")
                                .long("ip")
                                .default_value("127.0.0.1")
                                .takes_value(true),
                            Arg::with_name("security-key")
                                .help("Assign this key instead of randomly generated.")
                                .long("key")
                                .short("k")
                                .takes_value(true),
                            Arg::with_name("config-path")
                                .help("Set path for new config.")
                                .default_value("deployconf.yaml")
                                .short("P")
                                .long("config")
                                .takes_value(true),
                            Arg::with_name("overwrite")
                                .help("Forcibly overwrites current config if exists.")
                                .short("o")
                                .long("overwrite")
                        ]
                    )
            ]
        );
    let m = app.get_matches();

    debug!("Debug messages are enabled");

    if let Some(m) = m.subcommand_matches("new") {
        new::create_new_config(m);
    } else if let Some(m) = m.subcommand_matches("run") {
        if let Some(e) = run::run_server(m).await.err() {
            error!("{}", e);
        }
    } else {
        println!("{}", m.usage());
    }
}
