use std::{rc::Rc, sync::Arc, time::Instant};

use crate::{ArcString, RcString, RcVec};

const N: usize = 1_000_000;

#[test]
fn rc_vec_faster() {
    let strings: Vec<Vec<u8>> = (0..N).map(|i| i.to_string().as_bytes().to_vec()).collect();

    let start = Instant::now();
    for s in &strings {
        let mut s = RcVec::from(s.as_slice());
        s.push(1);
    }
    let lib_dur = (Instant::now() - start).as_secs_f32();
    println!("RcVec: {}", lib_dur);

    let start = Instant::now();
    for s in &strings {
        let mut s = Rc::new(Vec::from(s.as_slice()));
        Rc::make_mut(&mut s).push(1);
    }
    let std_dur = (Instant::now() - start).as_secs_f32();
    println!("Rc<Vec>: {}", std_dur);

    let faster = midpoint_diff(lib_dur, std_dur);
    println!("RcVec is {:.1}% faster than Rc<Vec>", faster * 100.0);

    assert!(lib_dur < std_dur);
}

#[test]
fn rrc_string_faster() {
    let strings: Vec<String> = (0..N).map(|i| i.to_string()).collect();

    let start = Instant::now();
    for s in &strings {
        let mut s = RcString::from(s.as_str());
        s.push('r');
    }
    let lib_dur = (Instant::now() - start).as_secs_f32();
    println!("RcString: {}", lib_dur);

    let start = Instant::now();
    for s in &strings {
        let mut s = Rc::new(String::from(s.as_str()));
        Rc::make_mut(&mut s).push('r');
    }
    let std_dur = (Instant::now() - start).as_secs_f32();
    println!("Rc<String>: {}", std_dur);

    let faster = midpoint_diff(lib_dur, std_dur);
    println!("RcString is {:.1}% faster than Rc<String>", faster * 100.0);

    assert!(lib_dur < std_dur);
}

#[test]
fn arc_string_faster() {
    let strings: Vec<String> = (0..N).map(|i| i.to_string()).collect();

    let start = Instant::now();
    for s in &strings {
        let mut s = ArcString::from(s.as_str());
        s.push('r');
    }
    let lib_dur = (Instant::now() - start).as_secs_f32();
    println!("ArcString: {}", lib_dur);

    let start = Instant::now();
    for s in &strings {
        let mut s = Arc::new(String::from(s.as_str()));
        Arc::make_mut(&mut s).push('r');
    }
    let std_dur = (Instant::now() - start).as_secs_f32();
    println!("Arc<String>: {}", std_dur);

    let faster = midpoint_diff(lib_dur, std_dur);
    println!(
        "ArcString is {:.1}% faster than Arc<String>",
        faster * 100.0
    );

    assert!(lib_dur < std_dur);
}

fn midpoint_diff(smaller: f32, bigger: f32) -> f32 {
    (1.0 - smaller / bigger + bigger / smaller - 1.0) / 2.0
}
