use core::fmt;
use std::{
    alloc::{alloc, Layout},
    borrow::{Borrow, Cow},
    cell::Cell,
    cmp::Ordering,
    ffi::{OsStr, OsString},
    hash::{Hash, Hasher},
    iter::FromIterator,
    mem::size_of,
    ops::Deref,
    ptr::{self, NonNull},
    rc::Rc,
    slice,
    sync::Arc,
};

/// A shared, reference-counted resizable buffer with copy-on-write behavior
pub struct RcVec<T> {
    inner: NonNull<Inner<T>>,
    len: usize,
    capacity: usize,
}

#[repr(C)]
struct Inner<T> {
    count: Cell<usize>,
    slice: [T],
}

impl<T> Default for RcVec<T> {
    fn default() -> Self {
        Self::new()
    }
}

impl<T> RcVec<T> {
    /// Create a new vector
    pub fn new() -> Self {
        Self::from([].as_ref())
    }
    /// Create a new vector with the given capacity preallocated
    pub fn with_capacity(capacity: usize) -> Self {
        RcVec {
            inner: unsafe { alloc_inner(&[], capacity) },
            len: 0,
            capacity,
        }
    }
    #[inline(always)]
    fn count(&self) -> &Cell<usize> {
        unsafe { &(*self.inner.as_ptr()).count }
    }
    /// Get a reference to the underlying slice
    pub fn as_slice(&self) -> &[T] {
        self
    }
    /// Push a value onto to the vector
    ///
    /// This copies the buffer if any other references exist.
    pub fn push(&mut self, item: T) {
        // Determine new capacity
        let new_len = self.len + 1;
        let new_capacity = if new_len > self.capacity {
            new_len.max(self.capacity * 2)
        } else {
            self.capacity
        };
        // Make a new inner if there is more than one clone
        self.ensure_uniqueness(new_capacity);
        // Write the item
        unsafe {
            let slice_ptr = (*self.inner.as_ptr()).slice.as_mut_ptr();
            let slice_end_ptr = slice_ptr.wrapping_add(self.len());
            ptr::write(slice_end_ptr, item);
        }
        // Increment length
        self.len = new_len;
    }
    /// Pop an item from the vector
    ///
    /// This copies the buffer if any other references exist.
    pub fn pop(&mut self) -> Option<T> {
        if self.len() == 0 {
            return None;
        }
        let item = unsafe {
            let slice_ptr = (*self.inner.as_ptr()).slice.as_mut_ptr();
            let last_ptr = slice_ptr.wrapping_add(self.len() - 1);
            self.ensure_uniqueness(self.capacity);
            ptr::read(last_ptr)
        };
        self.len -= 1;
        Some(item)
    }
    /// Clear all items from the vector
    ///
    /// This copies the buffer if any other references exist.
    pub fn clear(&mut self) {
        self.ensure_uniqueness(self.capacity);
        self.len = 0;
    }
    fn ensure_uniqueness(&mut self, target_capacity: usize) {
        let old_count = self.count().get();
        if old_count > 1 {
            self.count().set(self.count().get() - 1);
        }
        if old_count > 1 || self.capacity < target_capacity {
            self.inner = unsafe { alloc_inner(self, target_capacity) };
            self.capacity = target_capacity;
        }
    }
}

unsafe fn alloc_inner<T>(data: &[T], capacity: usize) -> NonNull<Inner<T>> {
    let size = size_of::<Cell<usize>>() + capacity;
    let layout = Layout::from_size_align(size, size.next_power_of_two()).unwrap();
    let buffer = alloc(layout);
    ptr::write(buffer as *mut Cell<usize>, Cell::new(1));
    ptr::copy_nonoverlapping(
        data.as_ptr() as *const u8,
        buffer.wrapping_add(size_of::<Cell<usize>>()),
        data.len() * size_of::<T>(),
    );
    let slice = slice::from_raw_parts_mut(buffer as *mut (), capacity);
    NonNull::new(slice as *mut [()] as *mut Inner<T>).unwrap()
}

impl<'a, T> From<&'a [T]> for RcVec<T> {
    fn from(data: &'a [T]) -> Self {
        Self {
            inner: unsafe { alloc_inner(data, data.len()) },
            len: data.len(),
            capacity: data.len(),
        }
    }
}

impl<T> From<Vec<T>> for RcVec<T> {
    fn from(v: Vec<T>) -> Self {
        Self::from(v.as_slice())
    }
}

impl<T> Clone for RcVec<T> {
    fn clone(&self) -> Self {
        let count = self.count();
        count.set(count.get() + 1);
        RcVec {
            inner: self.inner,
            len: self.len,
            capacity: self.capacity,
        }
    }
}

impl<T> Drop for RcVec<T> {
    fn drop(&mut self) {
        let count = self.count();
        count.set(count.get() - 1);
        if count.get() == 0 {
            unsafe {
                ptr::drop_in_place(self.inner.as_ptr());
            }
        }
    }
}

impl<T> fmt::Debug for RcVec<T>
where
    T: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

impl<T> Deref for RcVec<T> {
    type Target = [T];
    fn deref(&self) -> &Self::Target {
        unsafe { &(*self.inner.as_ptr()).slice[..self.len] }
    }
}

impl<T> PartialEq for RcVec<T>
where
    T: PartialEq,
{
    fn eq(&self, other: &Self) -> bool {
        (**self) == (**other)
    }
}

impl<T> Eq for RcVec<T> where T: Eq {}

impl<T> PartialOrd for RcVec<T>
where
    T: PartialOrd,
{
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        (**self).partial_cmp(other)
    }
}

impl<T> Ord for RcVec<T>
where
    T: Ord,
{
    fn cmp(&self, other: &Self) -> Ordering {
        (**self).cmp(other)
    }
}

impl<T> Hash for RcVec<T>
where
    T: Hash,
{
    fn hash<H>(&self, state: &mut H)
    where
        H: Hasher,
    {
        (**self).hash(state);
    }
}

impl<T> Borrow<[T]> for RcVec<T> {
    fn borrow(&self) -> &[T] {
        self
    }
}

impl<T> AsRef<[T]> for RcVec<T> {
    fn as_ref(&self) -> &[T] {
        self
    }
}

impl<T> PartialEq<str> for RcVec<T> {
    fn eq(&self, other: &str) -> bool {
        self == other
    }
}

impl<T> PartialEq<String> for RcVec<T> {
    fn eq(&self, other: &String) -> bool {
        self == other
    }
}

impl<'a, T> PartialEq<Cow<'a, [T]>> for RcVec<T>
where
    [T]: ToOwned,
{
    fn eq(&self, other: &Cow<'a, [T]>) -> bool {
        self == other
    }
}

impl<T> PartialEq<OsStr> for RcVec<T> {
    fn eq(&self, other: &OsStr) -> bool {
        self == other
    }
}

impl<T> PartialEq<OsString> for RcVec<T> {
    fn eq(&self, other: &OsString) -> bool {
        self == other
    }
}

impl<T> PartialEq<Rc<[T]>> for RcVec<T> {
    fn eq(&self, other: &Rc<[T]>) -> bool {
        self == other
    }
}

impl<T> PartialEq<Arc<[T]>> for RcVec<T> {
    fn eq(&self, other: &Arc<[T]>) -> bool {
        self == other
    }
}

impl<T> FromIterator<T> for RcVec<T> {
    fn from_iter<I>(iter: I) -> Self
    where
        I: IntoIterator<Item = T>,
    {
        let iter = iter.into_iter();
        let (min, max) = iter.size_hint();
        let mut vec = RcVec::with_capacity(max.unwrap_or(min));
        for item in iter {
            vec.push(item);
        }
        vec
    }
}
