use core::fmt;
use std::{
    alloc::{alloc, Layout},
    borrow::{Borrow, Cow},
    cell::Cell,
    cmp::Ordering,
    ffi::{OsStr, OsString},
    hash::{Hash, Hasher},
    mem::size_of,
    ops::Deref,
    path::Path,
    ptr::{self, NonNull},
    rc::Rc,
    slice,
    sync::Arc,
};

pub use crate::arc_string::ArcString;

/// A shared, reference-counted string buffer with copy-on-write behavior
pub struct RcString {
    inner: NonNull<Inner>,
    len: usize,
    capacity: usize,
}

#[repr(C)]
struct Inner {
    count: Cell<usize>,
    string: str,
}

impl Default for RcString {
    fn default() -> Self {
        Self::new()
    }
}

impl RcString {
    /// Create a new string
    pub fn new() -> Self {
        Self::from("")
    }
    /// Create a new string with the given capacity preallocated
    pub fn with_capacity(capacity: usize) -> Self {
        RcString {
            inner: unsafe { alloc_inner("", capacity) },
            len: 0,
            capacity,
        }
    }
    #[inline(always)]
    fn count(&self) -> &Cell<usize> {
        unsafe { &(*self.inner.as_ptr()).count }
    }
    /// Get a reference to the underlying `str`
    pub fn as_str(&self) -> &str {
        self
    }
    /// Push a string onto the string
    ///
    /// This copies the string buffer if any other references exist.
    pub fn push(&mut self, ch: char) {
        self.push_str(ch.encode_utf8(&mut [0u8; 4]));
    }
    /// Push a string onto to the string
    ///
    /// This copies the string buffer if any other references exist.
    pub fn push_str(&mut self, string: &str) {
        // Determine new capacity
        let new_len = self.len + string.len();
        let new_capacity = if new_len > self.capacity {
            new_len.max(self.capacity * 2)
        } else {
            self.capacity
        };
        // Make a new inner if there is more than one clone
        self.ensure_uniqueness(new_capacity);
        // Write the character
        unsafe {
            let string_ptr = (*self.inner.as_ptr()).string.as_mut_ptr();
            let string_end_ptr = string_ptr.wrapping_add(self.len);
            ptr::copy_nonoverlapping(string.as_ptr(), string_end_ptr, string.len());
        }
        // Increment length
        self.len = new_len;
    }
    /// Pop a character from the string
    ///
    /// This copies the string buffer if any other references exist.
    pub fn pop(&mut self) -> Option<char> {
        let ch = self.chars().rev().next()?;
        self.ensure_uniqueness(self.capacity);
        self.len -= ch.len_utf8();
        Some(ch)
    }
    /// Clear all characters from the string
    ///
    /// This copies the string buffer if any other references exist.
    pub fn clear(&mut self) {
        self.ensure_uniqueness(self.capacity);
        self.len = 0;
    }
    fn ensure_uniqueness(&mut self, target_capacity: usize) {
        let old_count = self.count().get();
        if old_count > 1 {
            self.count().set(self.count().get() - 1);
        }
        if old_count > 1 || self.capacity < target_capacity {
            self.inner = unsafe { alloc_inner(self, target_capacity) };
            self.capacity = target_capacity;
        }
    }
}

/// Allocate an RcString's Inner from the given string and capacity
///
/// # Safety
/// `capacity` must be greater or equal to `s.len()`
unsafe fn alloc_inner(s: &str, capacity: usize) -> NonNull<Inner> {
    let size = size_of::<Cell<usize>>() + capacity;
    let layout = Layout::from_size_align(size, size.next_power_of_two()).unwrap();
    let buffer = alloc(layout);
    ptr::write(buffer as *mut Cell<usize>, Cell::new(1));
    ptr::copy_nonoverlapping(
        s.as_ptr(),
        buffer.wrapping_add(size_of::<Cell<usize>>()),
        s.len(),
    );
    let slice = slice::from_raw_parts_mut(buffer as *mut (), capacity);
    NonNull::new(slice as *mut [()] as *mut Inner).unwrap()
}

impl<'a> From<&'a str> for RcString {
    fn from(s: &'a str) -> Self {
        Self {
            inner: unsafe { alloc_inner(s, s.len()) },
            len: s.len(),
            capacity: s.len(),
        }
    }
}

impl From<String> for RcString {
    fn from(s: String) -> Self {
        Self::from(s.as_str())
    }
}

impl Clone for RcString {
    fn clone(&self) -> Self {
        let count = self.count();
        count.set(count.get() + 1);
        RcString {
            inner: self.inner,
            len: self.len,
            capacity: self.capacity,
        }
    }
}

impl Drop for RcString {
    fn drop(&mut self) {
        let count = self.count();
        count.set(count.get() - 1);
        if count.get() == 0 {
            unsafe {
                ptr::drop_in_place(self.inner.as_ptr());
            }
        }
    }
}

impl fmt::Debug for RcString {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

impl fmt::Display for RcString {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

impl Deref for RcString {
    type Target = str;
    fn deref(&self) -> &Self::Target {
        unsafe { &(*self.inner.as_ptr()).string[..self.len] }
    }
}

impl PartialEq for RcString {
    fn eq(&self, other: &Self) -> bool {
        (**self) == (**other)
    }
}

impl Eq for RcString {}

impl PartialOrd for RcString {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

impl Ord for RcString {
    fn cmp(&self, other: &Self) -> Ordering {
        (**self).cmp(other)
    }
}

impl Hash for RcString {
    fn hash<H>(&self, state: &mut H)
    where
        H: Hasher,
    {
        (**self).hash(state);
    }
}

impl Borrow<str> for RcString {
    fn borrow(&self) -> &str {
        self
    }
}

impl AsRef<str> for RcString {
    fn as_ref(&self) -> &str {
        self
    }
}

impl AsRef<[u8]> for RcString {
    fn as_ref(&self) -> &[u8] {
        (**self).as_ref()
    }
}

impl AsRef<OsStr> for RcString {
    fn as_ref(&self) -> &OsStr {
        (**self).as_ref()
    }
}

impl AsRef<Path> for RcString {
    fn as_ref(&self) -> &Path {
        (**self).as_ref()
    }
}

impl PartialEq<str> for RcString {
    fn eq(&self, other: &str) -> bool {
        self == other
    }
}

impl PartialEq<String> for RcString {
    fn eq(&self, other: &String) -> bool {
        self == other
    }
}

impl<'a> PartialEq<Cow<'a, str>> for RcString {
    fn eq(&self, other: &Cow<'a, str>) -> bool {
        self == other
    }
}

impl PartialEq<OsStr> for RcString {
    fn eq(&self, other: &OsStr) -> bool {
        self == other
    }
}

impl PartialEq<OsString> for RcString {
    fn eq(&self, other: &OsString) -> bool {
        self == other
    }
}

impl PartialEq<Rc<str>> for RcString {
    fn eq(&self, other: &Rc<str>) -> bool {
        self == other
    }
}

impl PartialEq<Arc<str>> for RcString {
    fn eq(&self, other: &Arc<str>) -> bool {
        self == other
    }
}

impl PartialEq<ArcString> for RcString {
    fn eq(&self, other: &ArcString) -> bool {
        self == other
    }
}

impl PartialOrd<str> for RcString {
    fn partial_cmp(&self, other: &str) -> Option<Ordering> {
        (**self).partial_cmp(other)
    }
}

impl PartialOrd<String> for RcString {
    fn partial_cmp(&self, other: &String) -> Option<Ordering> {
        (**self).partial_cmp(other)
    }
}
