use core::fmt;
use std::{
    alloc::{alloc, Layout},
    borrow::{Borrow, Cow},
    cell::Cell,
    cmp::Ordering,
    ffi::{OsStr, OsString},
    hash::{Hash, Hasher},
    mem::size_of,
    ops::Deref,
    path::Path,
    ptr::{self, NonNull},
    rc::Rc,
    slice,
    sync::{
        atomic::{self, AtomicUsize},
        Arc,
    },
};

use crate::RcString;

/// A thread-safe, shared, reference-counted string buffer with copy-on-write behavior
///
/// `ArcString` is to `RcString` as `Arc<T>` is to `Rc<T>`.
pub struct ArcString {
    inner: NonNull<Inner>,
    len: usize,
    capacity: usize,
}

unsafe impl Send for ArcString {}
unsafe impl Sync for ArcString {}

#[repr(C)]
struct Inner {
    count: AtomicUsize,
    string: str,
}

impl Default for ArcString {
    fn default() -> Self {
        Self::new()
    }
}

impl ArcString {
    /// Create a new string
    pub fn new() -> Self {
        Self::from("")
    }
    /// Create a new string with the given capacity preallocated
    pub fn with_capacity(capacity: usize) -> Self {
        ArcString {
            inner: unsafe { alloc_inner("", capacity) },
            len: 0,
            capacity,
        }
    }
    #[inline(always)]
    fn count(&self) -> &AtomicUsize {
        unsafe { &(*self.inner.as_ptr()).count }
    }
    /// Get a reference to the underlying `str`
    pub fn as_str(&self) -> &str {
        self
    }
    /// Push a string onto the string
    ///
    /// This copies the string buffer if any other references exist.
    pub fn push(&mut self, ch: char) {
        self.push_str(ch.encode_utf8(&mut [0u8; 4]));
    }
    /// Push a string onto to the string
    ///
    /// This copies the string buffer if any other references exist.
    pub fn push_str(&mut self, string: &str) {
        // Determine new capacity
        let new_len = self.len + string.len();
        let new_capacity = if new_len > self.capacity {
            new_len.max(self.capacity * 2)
        } else {
            self.capacity
        };
        // Make a new inner if there is more than one clone
        self.ensure_uniqueness(new_capacity);
        // Write the character
        unsafe {
            let string_ptr = (*self.inner.as_ptr()).string.as_mut_ptr();
            let string_end_ptr = string_ptr.wrapping_add(self.len);
            ptr::copy_nonoverlapping(string.as_ptr(), string_end_ptr, string.len());
        }
        // Increment length
        self.len = new_len;
    }
    /// Pop a character from the string
    ///
    /// This copies the string buffer if any other references exist.
    pub fn pop(&mut self) -> Option<char> {
        let ch = self.chars().rev().next()?;
        self.ensure_uniqueness(self.capacity);
        self.len -= ch.len_utf8();
        Some(ch)
    }
    /// Clear all characters from the string
    ///
    /// This copies the string buffer if any other references exist.
    pub fn clear(&mut self) {
        self.ensure_uniqueness(self.capacity);
        self.len = 0;
    }
    fn ensure_uniqueness(&mut self, target_capacity: usize) {
        let old_count = self
            .count()
            .fetch_update(
                atomic::Ordering::SeqCst,
                atomic::Ordering::SeqCst,
                |count| Some(if count > 1 { count - 1 } else { count }),
            )
            .unwrap();
        if old_count > 1 || self.capacity < target_capacity {
            self.inner = unsafe { alloc_inner(self, target_capacity) };
            self.capacity = target_capacity;
        }
    }
}
unsafe fn alloc_inner(s: &str, capacity: usize) -> NonNull<Inner> {
    let size = size_of::<Cell<usize>>() + capacity;
    let layout = Layout::from_size_align(size, size.next_power_of_two()).unwrap();
    let buffer = alloc(layout);
    ptr::write(buffer as *mut Cell<usize>, Cell::new(1));
    ptr::copy_nonoverlapping(
        s.as_ptr(),
        buffer.wrapping_add(size_of::<Cell<usize>>()),
        s.len(),
    );
    let slice = slice::from_raw_parts_mut(buffer as *mut (), capacity);
    NonNull::new(slice as *mut [()] as *mut Inner).unwrap()
}

impl<'a> From<&'a str> for ArcString {
    fn from(s: &'a str) -> Self {
        Self {
            inner: unsafe { alloc_inner(s, s.len()) },
            len: s.len(),
            capacity: s.len(),
        }
    }
}

impl From<String> for ArcString {
    fn from(s: String) -> Self {
        Self::from(s.as_str())
    }
}

impl Clone for ArcString {
    fn clone(&self) -> Self {
        self.count().fetch_add(1, atomic::Ordering::SeqCst);
        ArcString {
            inner: self.inner,
            len: self.len,
            capacity: self.capacity,
        }
    }
}

impl Drop for ArcString {
    fn drop(&mut self) {
        let old_count = self.count().fetch_sub(1, atomic::Ordering::SeqCst);
        if old_count == 1 {
            unsafe {
                ptr::drop_in_place(self.inner.as_ptr());
            }
        }
    }
}

impl fmt::Debug for ArcString {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

impl fmt::Display for ArcString {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

impl Deref for ArcString {
    type Target = str;
    fn deref(&self) -> &Self::Target {
        unsafe { &(*self.inner.as_ptr()).string[..self.len] }
    }
}

impl PartialEq for ArcString {
    fn eq(&self, other: &Self) -> bool {
        (**self) == (**other)
    }
}

impl Eq for ArcString {}

impl PartialOrd for ArcString {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

impl Ord for ArcString {
    fn cmp(&self, other: &Self) -> Ordering {
        (**self).cmp(other)
    }
}

impl Hash for ArcString {
    fn hash<H>(&self, state: &mut H)
    where
        H: Hasher,
    {
        (**self).hash(state);
    }
}

impl Borrow<str> for ArcString {
    fn borrow(&self) -> &str {
        self
    }
}

impl AsRef<str> for ArcString {
    fn as_ref(&self) -> &str {
        self
    }
}

impl AsRef<[u8]> for ArcString {
    fn as_ref(&self) -> &[u8] {
        (**self).as_ref()
    }
}

impl AsRef<OsStr> for ArcString {
    fn as_ref(&self) -> &OsStr {
        (**self).as_ref()
    }
}

impl AsRef<Path> for ArcString {
    fn as_ref(&self) -> &Path {
        (**self).as_ref()
    }
}

impl PartialEq<str> for ArcString {
    fn eq(&self, other: &str) -> bool {
        self == other
    }
}

impl PartialEq<String> for ArcString {
    fn eq(&self, other: &String) -> bool {
        self == other
    }
}

impl<'a> PartialEq<Cow<'a, str>> for ArcString {
    fn eq(&self, other: &Cow<'a, str>) -> bool {
        self == other
    }
}

impl PartialEq<OsStr> for ArcString {
    fn eq(&self, other: &OsStr) -> bool {
        self == other
    }
}

impl PartialEq<OsString> for ArcString {
    fn eq(&self, other: &OsString) -> bool {
        self == other
    }
}

impl PartialEq<Rc<str>> for ArcString {
    fn eq(&self, other: &Rc<str>) -> bool {
        self == other
    }
}

impl PartialEq<Arc<str>> for ArcString {
    fn eq(&self, other: &Arc<str>) -> bool {
        self == other
    }
}

impl PartialEq<RcString> for ArcString {
    fn eq(&self, other: &RcString) -> bool {
        self == other
    }
}

impl PartialOrd<str> for ArcString {
    fn partial_cmp(&self, other: &str) -> Option<Ordering> {
        (**self).partial_cmp(other)
    }
}

impl PartialOrd<String> for ArcString {
    fn partial_cmp(&self, other: &String) -> Option<Ordering> {
        (**self).partial_cmp(other)
    }
}
