use std::ops::{Deref, DerefMut};
use std::str::FromStr;
use bson::Bson;
use bson::spec::BinarySubtype;
use chrono::Utc;
use serde::{Deserializer, Serializer};
use serde::de::Error;
use crate::types::BINARY_SUBTYPE_TIME_UTC;


#[derive(Clone, Copy, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub struct TimeUtc {
    pub inner: chrono::NaiveTime,
}

impl From<chrono::NaiveTime> for TimeUtc {
    fn from(arg: chrono::NaiveTime) -> Self {
        Self {
            inner: arg
        }
    }
}

impl From<&chrono::NaiveTime> for TimeUtc {
    fn from(arg: &chrono::NaiveTime) -> Self {
        Self {
            inner: arg.clone()
        }
    }
}

impl serde::Serialize for TimeUtc {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: Serializer {
        let utc = self.inner.to_string();
        return bson::Binary {
            subtype: BinarySubtype::UserDefined(BINARY_SUBTYPE_TIME_UTC),
            bytes: utc.into_bytes(),
        }.serialize(serializer);
    }
}

impl<'de> serde::Deserialize<'de> for TimeUtc {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: Deserializer<'de> {
        match Bson::deserialize(deserializer)? {
            Bson::String(s) => {
                return Ok(Self {
                    inner: chrono::NaiveTime::from_str(&s).or_else(|e| Err(D::Error::custom(e.to_string())))?,
                });
            }
            Bson::Binary(data) => {
                let s = String::from_utf8(data.bytes).unwrap_or_default();
                return Ok(Self {
                    inner: chrono::NaiveTime::from_str(&s).or_else(|e| Err(D::Error::custom(e.to_string())))?,
                });
            }
            _ => {
                Err(D::Error::custom("deserialize un supported bson type!"))
            }
        }
    }
}

impl std::fmt::Display for TimeUtc {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl std::fmt::Debug for TimeUtc {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl Deref for TimeUtc {
    type Target = chrono::NaiveTime;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

impl DerefMut for TimeUtc {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.inner
    }
}

impl TimeUtc {
    /// Returns a [`DateTime`] which corresponds to the current date and time.
    pub fn now() -> TimeUtc {
        let utc = Utc::now();
        let dt = bson::DateTime::from_millis(utc.timestamp_millis());
        Self {
            inner: dt.to_chrono().with_timezone(&Utc).time()
        }
    }

    /// create from str
    pub fn from_str(arg: &str) -> Result<Self, crate::error::Error> {
        let inner = chrono::NaiveTime::from_str(arg)?;
        Ok(Self {
            inner: inner
        })
    }
}

#[cfg(test)]
mod test {
    use crate::types::TimeUtc;

    #[test]
    fn test_ser_de() {
        let b = TimeUtc::now();
        let bsons = bson::to_bson(&b).unwrap();
        let b_de: TimeUtc = bson::from_bson(bsons).unwrap();
        assert_eq!(b, b_de);
    }
}