pub const BINARY_SUBTYPE_JSON: u8 = 0x90;
pub const BINARY_SUBTYPE_DECIMAL: u8 = 0x91;
pub const BINARY_SUBTYPE_DATETIME_UTC: u8 = 0x92;
pub const BINARY_SUBTYPE_DATE_UTC: u8 = 0x93;
pub const BINARY_SUBTYPE_DATE_LOCAL: u8 = 0x94;
pub const BINARY_SUBTYPE_TIME_UTC: u8 = 0x95;
pub const BINARY_SUBTYPE_TIME_LOCAL: u8 = 0x96;
pub const BINARY_SUBTYPE_TIMESTAMP_Z: u8 = 0x97;


pub mod json;

use std::fmt::Formatter;
use bson::Bson;
use bson::spec::BinarySubtype;
pub use json::*;

pub mod uuids;

pub use uuids::*;

pub mod bytes;

pub use bytes::*;

pub mod datetime_native;

pub use datetime_native::*;

pub mod datetime_utc;

pub use datetime_utc::*;

pub mod date_native;

pub use date_native::*;

pub mod date_utc;

pub use date_utc::*;

pub mod time_native;

pub use time_native::*;

pub mod time_utc;

pub use time_utc::*;

pub mod decimal;

pub use decimal::*;

pub mod timestamp;

pub use timestamp::*;

pub mod timestamp_z;

pub use timestamp_z::*;


pub trait Format {
    fn do_format(&self) -> String;
}


impl Format for Bson {
    fn do_format(&self) -> String {
        match self {
            Bson::Double(d) => { format!("{}", d) }
            Bson::String(s) => { format!("\"{}\"", s) }
            Bson::Array(arr) => {
                arr.do_format()
            }
            Bson::Document(d) => {
                let mut buf = String::new();
                buf.push_str("{");
                for (k, v) in d {
                    buf.push_str("\"");
                    buf.push_str(k);
                    buf.push_str("\"");
                    buf.push_str(":");
                    buf.push_str(&v.do_format());
                    buf.push_str(",");
                }
                buf.pop();
                buf.push_str("}");
                buf
            }
            Bson::Boolean(b) => { format!("{}", b) }
            Bson::Null => { format!("null") }
            Bson::RegularExpression(j) => { format!("{:?}", j) }
            Bson::JavaScriptCode(j) => { format!("{:?}", j) }
            Bson::JavaScriptCodeWithScope(j) => { format!("{:?}", j) }
            Bson::Int32(i) => { format!("{}", i) }
            Bson::Int64(i) => { format!("{}", i) }
            Bson::Timestamp(s) => {
                format!("\"{}\"", Timestamp::from(s.clone()))
            }
            Bson::Binary(d) => {
                match d.subtype {
                    BinarySubtype::Generic => {
                        let bytes_len = d.bytes.len();
                        if bytes_len > 8192 {
                            //> 1kb
                            format!("bytes({})", d.bytes.len())
                        } else {
                            self.to_string()
                        }
                    }
                    BinarySubtype::Uuid => {
                        format!("\"{}\"", crate::types::Uuid::from(d))
                    }
                    BinarySubtype::UserDefined(type_id) => {
                        match type_id {
                            crate::types::BINARY_SUBTYPE_JSON => {
                                format!("{}", String::from_utf8(d.bytes.to_owned()).unwrap_or_default())
                            }
                            crate::types::BINARY_SUBTYPE_TIMESTAMP_Z => {
                                if let Ok(data) = bson::from_slice::<TimestampZ>(&d.bytes) {
                                    return format!("\"{}\"", data);
                                }
                                return String::new();
                            }
                            crate::types::BINARY_SUBTYPE_DECIMAL => {
                                if let Ok(data) = bson::from_slice::<Decimal>(&d.bytes) {
                                    return format!("{}", data);
                                }
                                return String::new();
                            }
                            crate::types::BINARY_SUBTYPE_DATETIME_UTC => {
                                if let Ok(data) = bson::from_slice::<DateTimeUtc>(&d.bytes) {
                                    return format!("\"{}\"", data);
                                }
                                return String::new();
                            }
                            crate::types::BINARY_SUBTYPE_DATE_LOCAL => {
                                if let Ok(data) = bson::from_slice::<DateNative>(&d.bytes) {
                                    return format!("\"{}\"", data);
                                }
                                return String::new();
                            }
                            crate::types::BINARY_SUBTYPE_DATE_UTC => {
                                if let Ok(data) = bson::from_slice::<DateUtc>(&d.bytes) {
                                    return format!("\"{}\"", data);
                                }
                                return String::new();
                            }
                            crate::types::BINARY_SUBTYPE_TIME_UTC => {
                                if let Ok(data) = bson::from_slice::<TimeUtc>(&d.bytes) {
                                    return format!("\"{}\"", data);
                                }
                                return String::new();
                            }
                            crate::types::BINARY_SUBTYPE_TIME_LOCAL => {
                                if let Ok(data) = bson::from_slice::<TimeNative>(&d.bytes) {
                                    return format!("\"{}\"", data);
                                }
                                return String::new();
                            }
                            _ => {
                                format!("un supported!")
                            }
                        }
                    }
                    _ => {
                        format!("un supported!")
                    }
                }
            }
            Bson::ObjectId(id) => {
                format!("\"{}\"", id)
            }
            Bson::DateTime(dt) => {
                format!("\"{}\"", DateTimeNative::from(dt.clone()))
            }
            Bson::Symbol(s) => { format!("{}", s) }
            Bson::Decimal128(d) => {
                format!("{}", d)
            }
            Bson::Undefined => {
                format!("{}", "Undefined")
            }
            Bson::MaxKey => { format!("{}", "MaxKey") }
            Bson::MinKey => { format!("{}", "MinKey") }
            Bson::DbPointer(p) => { format!("{:?}", p) }
        }
    }
}

impl Format for Vec<Bson> {
    fn do_format(&self) -> String {
        let mut buf = String::new();
        buf.push_str("[");
        for item in self {
            buf.push_str(&item.do_format());
            buf.push_str(",");
        }
        buf.pop();
        buf.push_str("]");
        buf
    }
}
