use std::ops::{Add, Deref, DerefMut};
use std::str::FromStr;
use std::time::SystemTime;
use bson::{Bson, DateTime};
use bson::spec::BinarySubtype;
use chrono::{Local, Utc};
use serde::{Deserializer, Serializer};
use serde::de::Error;

/// Rbatis DateTime
#[derive(Clone, Copy, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub struct DateTimeNative {
    pub inner: chrono::DateTime<Local>,
}

impl From<bson::DateTime> for DateTimeNative {
    fn from(arg: DateTime) -> Self {
        Self {
            inner: arg.to_chrono().with_timezone(&chrono::Local),
        }
    }
}

impl From<chrono::DateTime<Local>> for DateTimeNative {
    fn from(arg: chrono::DateTime<Local>) -> Self {
        DateTimeNative {
            inner: arg
        }
    }
}

impl From<&chrono::DateTime<Local>> for DateTimeNative {
    fn from(arg: &chrono::DateTime<Local>) -> Self {
        DateTimeNative {
            inner: arg.clone()
        }
    }
}

impl serde::Serialize for DateTimeNative {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: Serializer {
        return bson::DateTime::from_chrono(self.inner.clone()).serialize(serializer);
    }
}

impl<'de> serde::Deserialize<'de> for DateTimeNative {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: Deserializer<'de> {
        match Bson::deserialize(deserializer)? {
            Bson::DateTime(date) => {
                return Ok(Self {
                    inner: date.to_chrono().with_timezone(&chrono::Local),
                });
            }
            Bson::String(s) => {
                return Ok(Self {
                    inner: chrono::DateTime::<chrono::Local>::from_str(&s).or_else(|e| Err(D::Error::custom(e.to_string())))?,
                });
            }
            Bson::Binary(data) => {
                let s = String::from_utf8(data.bytes).unwrap_or_default();
                return Ok(Self {
                    inner: chrono::DateTime::<chrono::Local>::from_str(&s).or_else(|ex| Err(D::Error::custom(ex.to_string())))?,
                });
            }
            _ => {
                Err(D::Error::custom("deserialize un supported bson type!"))
            }
        }
    }
}

impl std::fmt::Display for DateTimeNative {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl std::fmt::Debug for DateTimeNative {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl Deref for DateTimeNative {
    type Target = chrono::DateTime<Local>;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

impl DerefMut for DateTimeNative {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.inner
    }
}


impl DateTimeNative {
    /// Returns a [`DateTime`] which corresponds to the current date and time.
    pub fn now() -> DateTimeNative {
        let dt = bson::DateTime::now();
        Self {
            inner: dt.to_chrono().with_timezone(&Local)
        }
    }

    /// create from str
    pub fn from_str(arg: &str) -> Result<Self, crate::error::Error> {
        let inner = chrono::DateTime::<Local>::from_str(arg)?;
        Ok(Self {
            inner: inner
        })
    }
}

#[cfg(test)]
mod test {
    use crate::types::{DateTimeNative, DateTimeUtc};

    #[test]
    fn test_native() {
        let dt = DateTimeNative::now();
        let s = bson::to_bson(&dt).unwrap();
        let dt_new: DateTimeNative = bson::from_bson(s).unwrap();
        println!("{},{}", dt.timestamp_millis(), dt_new.timestamp_millis());
        assert_eq!(dt, dt_new);
    }

    #[test]
    fn test_utc() {
        let dt = DateTimeUtc::now();
        let s = bson::to_bson(&dt).unwrap();
        let dt_new: DateTimeUtc = bson::from_bson(s).unwrap();
        println!("{},{}", dt.timestamp_millis(), dt_new.timestamp_millis());
        assert_eq!(dt, dt_new);
    }

    #[test]
    fn test_decode() {
        let s = bson::Bson::String("2021-10-31T05:09:16.167 UTC".to_string());
        let dt_new: DateTimeUtc = bson::from_bson(s).unwrap();
        println!("{},{}", 1635656956167i64, dt_new.timestamp_millis());
        assert_eq!(1635656956167i64, dt_new.timestamp_millis());
    }

    #[test]
    fn test_ser_de() {
        let b = DateTimeNative::now();
        let bsons = bson::to_bson(&b).unwrap();
        let b_de: DateTimeNative = bson::from_bson(bsons).unwrap();
        assert_eq!(b, b_de);
    }
}