use std::ops::{Deref, DerefMut};
use std::str::FromStr;
use bson::Bson;
use bson::spec::BinarySubtype;
use chrono::{Local, NaiveDate};
use serde::{Deserializer, Serializer};
use serde::de::Error;
use crate::types::BINARY_SUBTYPE_DATE_LOCAL;


/// Rust type              Postgres type(s)
/// chrono::NaiveDate      DATE
#[derive(Clone, Copy, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub struct DateNative {
    pub inner: chrono::NaiveDate,
}

impl From<chrono::NaiveDate> for DateNative {
    fn from(arg: chrono::NaiveDate) -> Self {
        DateNative {
            inner: arg
        }
    }
}

impl From<&chrono::NaiveDate> for DateNative {
    fn from(arg: &chrono::NaiveDate) -> Self {
        DateNative {
            inner: arg.clone()
        }
    }
}

impl serde::Serialize for DateNative {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: Serializer {
        let utc = self.inner.to_string();
        return bson::Binary {
            subtype: BinarySubtype::UserDefined(BINARY_SUBTYPE_DATE_LOCAL),
            bytes: utc.into_bytes(),
        }.serialize(serializer);
    }
}


impl<'de> serde::Deserialize<'de> for DateNative {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: Deserializer<'de> {
        match Bson::deserialize(deserializer)? {
            Bson::String(s) => {
                return Ok(Self {
                    inner: chrono::NaiveDate::from_str(&s).or_else(|e| Err(D::Error::custom(e.to_string())))?,
                });
            }
            Bson::Binary(data) => {
                let s = String::from_utf8(data.bytes).unwrap_or_default();
                return Ok(Self {
                    inner: chrono::NaiveDate::from_str(&s).or_else(|e| Err(D::Error::custom(e.to_string())))?,
                });
            }
            _ => {
                Err(D::Error::custom("deserialize un supported bson type!"))
            }
        }
    }
}

impl std::fmt::Display for DateNative {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl std::fmt::Debug for DateNative {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl Deref for DateNative {
    type Target = chrono::NaiveDate;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

impl DerefMut for DateNative {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.inner
    }
}


impl DateNative {
    /// Returns a [`DateTime`] which corresponds to the current date and time.
    pub fn now() -> DateNative {
        let utc = Local::now();
        Self {
            inner: utc.date().naive_local(),
        }
    }

    /// create from str
    pub fn from_str(arg: &str) -> Result<Self, crate::error::Error> {
        let inner = chrono::NaiveDate::from_str(arg)?;
        Ok(Self {
            inner: inner
        })
    }
}


#[cfg(test)]
mod test {
    use crate::types::DateNative;

    #[test]
    fn test_ser_de() {
        let b = DateNative::now();
        let bsons = bson::to_bson(&b).unwrap();
        let b_de: DateNative = bson::from_bson(bsons).unwrap();
        assert_eq!(b, b_de);
    }
}