use bson::Bson;
use bson::spec::BinarySubtype;
use sqlx_core::mysql::{MySql, MySqlArguments};
use sqlx_core::query::Query;
use crate::error::Error;
use crate::types::{DateNative, DateTimeNative, DateTimeUtc, DateUtc, Decimal, TimeNative, TimeUtc};

#[inline]
pub fn bind(t: Bson, mut q: Query<MySql, MySqlArguments>) -> crate::Result<Query<MySql, MySqlArguments>> {
    match t {
        Bson::String(s) => {
            q = q.bind(Some(s));
        }
        Bson::Null => {
            q = q.bind(Option::<String>::None);
        }
        Bson::Int32(n) => {
            q = q.bind(n);
        }
        Bson::Int64(n) => {
            q = q.bind(n);
        }
        Bson::Double(n) => {
            q = q.bind(n);
        }
        Bson::Boolean(b) => {
            q = q.bind(b);
        }
        Bson::Decimal128(d) => {
            q = q.bind(d.to_string());
        }
        Bson::Binary(d) => {
            match d.subtype {
                BinarySubtype::Generic => {
                    q = q.bind(d.bytes);
                }
                BinarySubtype::Uuid => {
                    q = q.bind(crate::types::Uuid::from(d).inner);
                }
                BinarySubtype::UserDefined(type_id) => {
                    match type_id {
                        crate::types::BINARY_SUBTYPE_JSON => {
                            q = q.bind(serde_json::from_slice::<serde_json::Value>(&d.bytes).unwrap_or_default());
                        }
                        crate::types::BINARY_SUBTYPE_DECIMAL => {
                            let data: Decimal = bson::from_slice(&d.bytes)?;
                            q = q.bind(data.inner);
                        }
                        crate::types::BINARY_SUBTYPE_DATETIME_UTC => {
                            let data: DateTimeUtc = bson::from_slice(&d.bytes)?;
                            q = q.bind(data.inner);
                        }
                        crate::types::BINARY_SUBTYPE_DATE_LOCAL => {
                            let data: DateNative = bson::from_slice(&d.bytes)?;
                            q = q.bind(data.inner);
                        }
                        crate::types::BINARY_SUBTYPE_DATE_UTC => {
                            let data: DateUtc = bson::from_slice(&d.bytes)?;
                            q = q.bind(data.inner);
                        }
                        crate::types::BINARY_SUBTYPE_TIME_UTC => {
                            let data: TimeUtc = bson::from_slice(&d.bytes)?;
                            q = q.bind(data.inner);
                        }
                        crate::types::BINARY_SUBTYPE_TIME_LOCAL => {
                            let data: TimeNative = bson::from_slice(&d.bytes)?;
                            q = q.bind(data.inner);
                        }
                        _ => {
                            return Err(Error::from("un supported bind type!"));
                        }
                    }
                }
                _ => {
                    return Err(Error::from("un supported bind type!"));
                }
            }
        }
        Bson::DateTime(d) => {
            q = q.bind(DateTimeNative::from(d).inner);
        }
        Bson::Timestamp(d) => {
            q = q.bind(crate::types::Timestamp::from(d).inner);
        }
        Bson::ObjectId(d) => {
            q = q.bind(d.to_string());
        }
        _ => {
            return crate::Result::Err(crate::Error::from("unsupported type!"));
        }
    }
    return Ok(q);
}