use std::alloc::Layout;
use std::ops::{Deref, DerefMut};
use bson::Bson;
use bson::spec::BinarySubtype;
use serde::{Deserializer, Serializer};
use serde::de::Error;
use sqlx_core::types::time;
use crate::types::BINARY_SUBTYPE_TIMESTAMP_Z;

/// Rbatis Timestamp
/// Rust type                Postgres type(s)
/// time::OffsetDateTime      TIMESTAMPTZ
#[derive(Clone, Copy, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub struct TimestampZ {
    pub inner: time::OffsetDateTime,
}

impl From<time::OffsetDateTime> for TimestampZ {
    fn from(arg: time::OffsetDateTime) -> Self {
        Self {
            inner: arg
        }
    }
}

impl From<&time::OffsetDateTime> for TimestampZ {
    fn from(arg: &time::OffsetDateTime) -> Self {
        Self {
            inner: arg.clone()
        }
    }
}

impl serde::Serialize for TimestampZ {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: Serializer {
        return bson::Binary {
            subtype: BinarySubtype::UserDefined(BINARY_SUBTYPE_TIMESTAMP_Z),
            bytes: self.inner.unix_timestamp().to_string().into_bytes(),
        }.serialize(serializer);
    }
}

impl<'de> serde::Deserialize<'de> for TimestampZ {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: Deserializer<'de> {
        match Bson::deserialize(deserializer)? {
            Bson::String(s) => {
                return Ok(Self {
                    inner: time::OffsetDateTime::parse(&s, "%F %T %z").or_else(|e| Err(D::Error::custom(e.to_string())))?,
                });
            }
            Bson::Int64(data) => {
                return Ok(Self::from_unix_timestamp(data));
            }
            Bson::Binary(data) => {
                let s = String::from_utf8(data.bytes).unwrap_or_default();
                let num = s.parse().unwrap_or_default();
                return Ok(Self {
                    inner: time::OffsetDateTime::from_unix_timestamp(num),
                });
            }
            _ => {
                Err(D::Error::custom("deserialize un supported bson type!"))
            }
        }
    }
}

impl TimestampZ {
    pub fn as_timestamp(arg: &bson::Timestamp) -> i64 {
        let upper = (arg.time.to_le() as u64) << 32;
        let lower = arg.increment.to_le() as u64;
        (upper | lower) as i64
    }

    pub fn from_le_i64(val: i64) -> bson::Timestamp {
        let ts = val.to_le();
        bson::Timestamp {
            time: ((ts as u64) >> 32) as u32,
            increment: (ts & 0xFFFF_FFFF) as u32,
        }
    }
}

impl From<bson::Timestamp> for TimestampZ {
    fn from(data: bson::Timestamp) -> Self {
        let offset = time::OffsetDateTime::from_unix_timestamp(TimestampZ::as_timestamp(&data));
        Self {
            inner: offset
        }
    }
}

impl std::fmt::Display for TimestampZ {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl std::fmt::Debug for TimestampZ {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl Deref for TimestampZ {
    type Target = time::OffsetDateTime;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

impl DerefMut for TimestampZ {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.inner
    }
}

impl TimestampZ {
    pub fn now() -> Self {
        Self {
            inner: time::OffsetDateTime::from_unix_timestamp(time::OffsetDateTime::now_local().unix_timestamp())
        }
    }

    /// create from str
    pub fn from_str(arg: &str) -> Result<Self, crate::error::Error> {
        let inner = time::OffsetDateTime::parse(arg, "%F %T %z")?;
        Ok(Self {
            inner: inner
        })
    }


    pub fn timestamp_millis(&self) -> i64 {
        self.inner.unix_timestamp()
    }

    pub fn from_unix_timestamp(arg: i64) -> TimestampZ {
        Self {
            inner: time::OffsetDateTime::from_unix_timestamp(arg)
        }
    }
}

#[cfg(test)]
mod test {
    use crate::types::TimestampZ;

    #[test]
    fn test_native() {
        let dt = TimestampZ::now();
        let s = bson::to_bson(&dt).unwrap();
        let dt_new: TimestampZ = bson::from_bson(s).unwrap();
        println!("{},{}", dt.timestamp_millis(), dt_new.timestamp_millis());
        assert_eq!(dt, dt_new);
    }

    #[test]
    fn test_ser_de() {
        let b = TimestampZ::now();
        let bsons = bson::to_bson(&b).unwrap();
        let b_de: TimestampZ = bson::from_bson(bsons).unwrap();
        assert_eq!(b, b_de);
    }
}
