use std::ops::{Deref, DerefMut};
use std::str::FromStr;
use bson::Bson;
use bson::spec::BinarySubtype;
use chrono::Utc;
use serde::{Deserializer, Serializer};
use serde::de::Error;
use crate::types::BINARY_SUBTYPE_DATE_UTC;


#[derive(Clone, Copy, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub struct DateUtc {
    pub inner: chrono::NaiveDate,
}

impl From<chrono::NaiveDate> for DateUtc {
    fn from(arg: chrono::NaiveDate) -> Self {
        DateUtc {
            inner: arg
        }
    }
}

impl From<&chrono::NaiveDate> for DateUtc {
    fn from(arg: &chrono::NaiveDate) -> Self {
        DateUtc {
            inner: arg.clone()
        }
    }
}

impl serde::Serialize for DateUtc {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: Serializer {
        let utc = self.inner.to_string();
        return bson::Binary {
            subtype: BinarySubtype::UserDefined(BINARY_SUBTYPE_DATE_UTC),
            bytes: utc.into_bytes(),
        }.serialize(serializer);
    }
}

impl<'de> serde::Deserialize<'de> for DateUtc {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: Deserializer<'de> {
        match Bson::deserialize(deserializer)? {
            Bson::String(s) => {
                return Ok(Self {
                    inner: chrono::NaiveDate::from_str(&s).or_else(|e| Err(D::Error::custom(e.to_string())))?,
                });
            }
            Bson::Binary(data) => {
                let s = String::from_utf8(data.bytes).unwrap_or_default();
                return Ok(Self {
                    inner: chrono::NaiveDate::from_str(&s).or_else(|e| Err(D::Error::custom(e.to_string())))?,
                });
            }
            _ => {
                Err(D::Error::custom("deserialize un supported bson type!"))
            }
        }
    }
}

impl std::fmt::Display for DateUtc {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl std::fmt::Debug for DateUtc {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl Deref for DateUtc {
    type Target = chrono::NaiveDate;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

impl DerefMut for DateUtc {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.inner
    }
}


impl DateUtc {
    /// Returns a [`DateTime`] which corresponds to the current date and time.
    pub fn now() -> DateUtc {
        let utc = Utc::now();
        Self {
            inner: utc.date().naive_local(),
        }
    }

    /// create from str
    pub fn from_str(arg: &str) -> Result<Self, crate::error::Error> {
        let inner = chrono::NaiveDate::from_str(arg)?;
        Ok(Self {
            inner: inner
        })
    }
}

#[cfg(test)]
mod test {
    use crate::types::DateUtc;

    #[test]
    fn test_ser_de() {
        let b = DateUtc::now();
        let bsons = bson::to_bson(&b).unwrap();
        let b_de: DateUtc = bson::from_bson(bsons).unwrap();
        assert_eq!(b, b_de);
    }
}