use std::fmt::Formatter;
use std::ops::{Deref, DerefMut};
use std::str::FromStr;
use bigdecimal_::{BigDecimal, ParseBigDecimalError};
use bson::Bson;
use bson::spec::BinarySubtype;
use serde::{Deserializer, Serialize, Serializer};
use serde::de::{Error, Visitor};
use crate::types::BINARY_SUBTYPE_DECIMAL;

/// Rbatis Decimal
#[derive(Clone, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub struct Decimal {
    pub inner: BigDecimal,
}

impl From<BigDecimal> for Decimal {
    fn from(arg: BigDecimal) -> Self {
        Self {
            inner: arg
        }
    }
}

impl From<&BigDecimal> for Decimal {
    fn from(arg: &BigDecimal) -> Self {
        Self {
            inner: arg.clone()
        }
    }
}

impl serde::Serialize for Decimal {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: Serializer {
        let b = bson::Binary {
            subtype: BinarySubtype::UserDefined(BINARY_SUBTYPE_DECIMAL),
            bytes: self.inner.to_string().into_bytes(),
        };
        return b.serialize(serializer);
    }
}

/// Decimal allow deserialize by an String or Binary
impl<'de> serde::Deserialize<'de> for Decimal {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: Deserializer<'de> {
        let bson = bson::Bson::deserialize(deserializer)?;
        return match bson {
            Bson::String(s) => {
                Ok(Self {
                    inner: BigDecimal::from_str(s.as_str()).unwrap_or_default(),
                })
            }
            Bson::Int32(s) => {
                Ok(Self {
                    inner: BigDecimal::from(s),
                })
            }
            Bson::Int64(s) => {
                Ok(Self {
                    inner: BigDecimal::from(s),
                })
            }
            Bson::Decimal128(s) => {
                Ok(Self {
                    inner: BigDecimal::from_str(&s.to_string()).unwrap_or_default(),
                })
            }
            Bson::Binary(data) => {
                let s = String::from_utf8(data.bytes).unwrap_or_else(|_| "0".to_string());
                Ok(Self {
                    inner: BigDecimal::from_str(s.as_str()).unwrap_or_default(),
                })
            }
            _ => {
                Err(D::Error::custom("deserialize unsupported bson type!"))
            }
        };
    }
}

impl std::fmt::Display for Decimal {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl std::fmt::Debug for Decimal {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.inner.fmt(f)
    }
}

impl Deref for Decimal {
    type Target = BigDecimal;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

impl DerefMut for Decimal {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.inner
    }
}


impl Decimal {
    pub fn from(s: &str) -> Self {
        let b = BigDecimal::from_str(s).unwrap_or_default();
        Self {
            inner: b
        }
    }

    /// create from str
    pub fn from_str(arg: &str) -> Result<Self, crate::error::Error> {
        let b = BigDecimal::from_str(arg)?;
        Ok(Self {
            inner: b
        })
    }
}

#[cfg(test)]
mod test {
    use crate::types::Decimal;

    #[test]
    fn test_ser_de() {
        let b = Decimal::from("1");
        let bsons = bson::to_bson(&b).unwrap();
        let b_de: Decimal = bson::from_bson(bsons).unwrap();
        assert_eq!(b, b_de);
    }
}