use std::ops::{Deref, DerefMut};
use bson::{Binary, Bson};
use bson::spec::BinarySubtype;
use serde::{Deserializer, Serializer};
use serde::de::Error;

/// Rbatis Bytes
#[derive(Clone, PartialEq, Debug)]
pub struct Bytes {
    pub inner: bson::Binary,
}

impl From<bson::Binary> for Bytes {
    fn from(arg: bson::Binary) -> Self {
        Self {
            inner: arg
        }
    }
}

impl From<&bson::Binary> for Bytes {
    fn from(arg: &bson::Binary) -> Self {
        Self {
            inner: arg.clone()
        }
    }
}

impl From<&[u8]> for Bytes {
    fn from(arg: &[u8]) -> Self {
        Self {
            inner: Binary {
                subtype: BinarySubtype::Generic,
                bytes: arg.to_vec(),
            }
        }
    }
}

impl From<Vec<u8>> for Bytes {
    fn from(arg: Vec<u8>) -> Self {
        Self {
            inner: Binary {
                subtype: BinarySubtype::Generic,
                bytes: arg,
            }
        }
    }
}

impl From<&Vec<u8>> for Bytes {
    fn from(arg: &Vec<u8>) -> Self {
        Self {
            inner: Binary {
                subtype: BinarySubtype::Generic,
                bytes: arg.clone(),
            }
        }
    }
}

impl serde::Serialize for Bytes {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: Serializer {
        return self.inner.serialize(serializer);
    }
}

impl<'de> serde::Deserialize<'de> for Bytes {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: Deserializer<'de> {
        let bson = Bson::deserialize(deserializer)?;
        match bson {
            Bson::Binary(data) => {
                return Ok(Bytes {
                    inner: data,
                });
            }
            _ => {
                Err(D::Error::custom("deserialize unsupported bson type!"))
            }
        }
    }
}

impl Bytes {
    pub fn new(arg: Vec<u8>) -> Bytes {
        Bytes {
            inner: Binary {
                subtype: BinarySubtype::Generic,
                bytes: arg,
            }
        }
    }
}

impl Deref for Bytes {
    type Target = Vec<u8>;

    fn deref(&self) -> &Self::Target {
        &self.inner.bytes
    }
}

impl DerefMut for Bytes {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.inner.bytes
    }
}


#[cfg(test)]
mod test {
    use crate::types::Bytes;

    #[test]
    fn test_ser_de() {
        let b = Bytes::from("111".as_bytes());
        let bsons = bson::to_bson(&b).unwrap();
        let b_de: Bytes = bson::from_bson(bsons).unwrap();
        assert_eq!(b, b_de);
    }
}