//! Types and traits for decoding values from the database.
use bson::Bson;
use serde::de::DeserializeOwned;

use crate::{Error};
use crate::types::Format;

/// decode json vec to an object
/// support decode types:
/// Bson,BigDecimal, i8..i64,u8..u64,bson::Int64(),bool,String
/// or object used bson macro object
pub fn decode<T: ?Sized>(mut datas: Vec<Bson>) -> Result<T, crate::Error>
    where
        T: DeserializeOwned,
{
    let mut js = Bson::Null;
    let mut type_name = std::any::type_name::<T>();
    //debug_mode feature print json_decode json data
    #[cfg(feature = "debug_mode")]
    println!("[rbatis] [debug_mode] => {}", datas.do_format());

    let is_array: Result<T, bson::de::Error> = bson::from_bson(bson::Bson::Array(vec![]));
    if is_array.is_ok() {
        //decode array
        js = Bson::Array(datas);
        return Ok(bson::from_bson(js)?);
    } else {
        //decode struct
        if datas.len() > 1 {
            return Result::Err(Error::from(format!("[rbatis] rows.rows_affected > 1,but decode one type ({})!", type_name)));
        }
        //single try decode
        if datas.is_empty() {
            return Ok(bson::from_bson::<T>(Bson::Null)?);
        }
        let mut v = None;
        let m = datas.remove(0);
        match &m {
            Bson::Document(d) => {
                if d.len() == 1 {
                    for (k, _v) in d {
                        v = Some(_v.clone());
                        break;
                    }
                }
            }
            _ => {}
        }
        let r = bson::from_bson::<T>(m);
        if r.is_err() {
            //try one
            let v = v.take().unwrap_or_default();
            return Ok(bson::from_bson::<T>(v)?);
        } else {
            return Ok(r.unwrap());
        }
    }
}


#[cfg(test)]
mod test {
    use std::collections::{BTreeMap, HashMap, HashSet, VecDeque};
    use bson::Bson;
    use bson::bson;
    use crate::decode::{decode};
    use crate::types::Json;

    #[test]
    fn test_decode_hashmap() {
        let m: HashMap<String, Bson> = decode(vec![bson!(
        {
        "a":"1",
        "b":2
        }
        )])
            .unwrap();
        println!("{:#?}", m);
    }

    #[test]
    fn test_decode_btree_map() {
        let m: BTreeMap<String, Bson> = decode(vec![bson!(
        {
        "a":"1",
        "b":2
        }
        )])
            .unwrap();
        println!("{:#?}", m);
    }
}
