//! Errorand Result types.
use std::any::type_name;
use std::error::Error as StdError;
use std::fmt::{self, Debug, Display};
use std::io;
use bigdecimal_::ParseBigDecimalError;
use chrono::ParseError;

use serde::{Deserialize, Deserializer};
use serde::de::Visitor;
use serde::ser::{Serialize, Serializer};
use sqlx_core::error::BoxDynError;

/// A specialized `Result` type for rbatis::core.
pub type Result<T> = std::result::Result<T, Error>;

/// A generic error that represents all the ways a method can fail inside of rbatis::core.
#[derive(Debug)]
#[non_exhaustive]
pub enum Error {
    /// Default Error
    E(String),
    Deserialize(String),
    Database(String),
}

impl Display for Error {
    // IntellijRust does not understand that [non_exhaustive] applies only for downstream crates
    // noinspection RsMatchCheck
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Error::E(error) => write!(f, "{}", error),
            Error::Deserialize(error) => write!(f, "{}", error),
            Error::Database(error) => write!(f, "{}", error.to_string()),
        }
    }
}

impl StdError for Error {}

impl From<io::Error> for Error {
    #[inline]
    fn from(err: io::Error) -> Self {
        Error::from(err.to_string())
    }
}

impl From<&str> for Error {
    fn from(arg: &str) -> Self {
        return Error::E(arg.to_string());
    }
}

impl From<std::string::String> for Error {
    fn from(arg: String) -> Self {
        return Error::E(arg);
    }
}

impl From<&dyn std::error::Error> for Error {
    fn from(arg: &dyn std::error::Error) -> Self {
        return Error::E(arg.to_string());
    }
}

impl From<rexpr::error::Error> for crate::Error {
    fn from(e: rexpr::error::Error) -> Self {
        crate::Error::E(format!("[rbatis]{}", e.to_string()))
    }
}

impl From<sqlx_core::error::BoxDynError> for crate::Error {
    fn from(arg: BoxDynError) -> Self {
        return crate::Error::Database(arg.to_string());
    }
}

impl From<sqlx_core::error::Error> for crate::Error {
    fn from(arg: sqlx_core::error::Error) -> Self {
        return crate::Error::Database(arg.to_string());
    }
}

impl From<chrono::ParseError> for crate::Error {
    fn from(arg: ParseError) -> Self {
        return crate::Error::E(arg.to_string());
    }
}

impl From<serde_json::Error> for crate::Error {
    fn from(arg: serde_json::Error) -> Self {
        return crate::Error::E(arg.to_string());
    }
}

impl From<time::ParseError> for crate::Error {
    fn from(arg: time::ParseError) -> Self {
        return crate::Error::E(arg.to_string());
    }
}

impl From<bson::ser::Error> for crate::Error {
    fn from(arg: bson::ser::Error) -> Self {
        return crate::Error::E(arg.to_string());
    }
}

impl From<bson::de::Error> for crate::Error {
    fn from(arg: bson::de::Error) -> Self {
        return crate::Error::E(arg.to_string());
    }
}

impl Clone for Error {
    fn clone(&self) -> Self {
        Error::from(self.to_string())
    }

    fn clone_from(&mut self, source: &Self) {
        *self = Self::from(source.to_string());
    }
}

// This is what #[derive(Serialize)] would generate.
impl Serialize for Error {
    fn serialize<S>(&self, serializer: S) -> std::result::Result<S::Ok, S::Error>
        where
            S: Serializer,
    {
        serializer.serialize_str(self.to_string().as_str())
    }
}

struct ErrorVisitor;

impl<'de> Visitor<'de> for ErrorVisitor {
    type Value = String;

    fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        formatter.write_str("a string")
    }

    fn visit_string<E>(self, v: String) -> std::result::Result<Self::Value, E>
        where
            E: std::error::Error,
    {
        Ok(v)
    }

    fn visit_str<E>(self, v: &str) -> std::result::Result<Self::Value, E>
        where
            E: std::error::Error,
    {
        Ok(v.to_string())
    }
}

impl<'de> Deserialize<'de> for Error {
    fn deserialize<D>(deserializer: D) -> std::result::Result<Self, D::Error>
        where
            D: Deserializer<'de>,
    {
        let r = deserializer.deserialize_string(ErrorVisitor)?;
        return Ok(Error::from(r));
    }
}

pub trait OptionToResult<T> {
    fn to_result(self, error_str: &str) -> Result<T>;
}

impl<T> OptionToResult<T> for Option<T> {
    fn to_result(self, error_str: &str) -> Result<T> {
        if self.is_some() {
            Ok(self.unwrap())
        } else {
            Err(Error::from(error_str))
        }
    }
}

impl From<py_sql::error::Error> for crate::Error {
    fn from(arg: py_sql::error::Error) -> Error {
        Error::E(arg.to_string())
    }
}

impl Into<py_sql::error::Error> for crate::Error {
    fn into(self) -> py_sql::error::Error {
        py_sql::error::Error::E(self.to_string())
    }
}

impl From<Error> for std::io::Error {
    fn from(arg: Error) -> Self {
        arg.into()
    }
}

impl From<ParseBigDecimalError> for crate::Error {
    fn from(arg: ParseBigDecimalError) -> Self {
        Self::E(arg.to_string())
    }
}

impl From<uuid::Error> for crate::Error {
    fn from(arg: uuid::Error) -> Self {
        Self::E(arg.to_string())
    }
}

#[test]
fn test_json_error() {
    let e = Error::from("fuck");
    let s = serde_json::to_string(&e).unwrap();
    println!("{}", s.as_str());
}
