use serde::de::Error;
use serde_json::Value;
use serde::{Deserializer, Serialize};

/// bytes
#[derive(Eq, PartialEq, Clone, Debug, serde::Serialize)]
pub struct RbJson {
    pub rb_json: serde_json::Value,
}


impl RbJson {
    pub fn from<T>(v: T) -> Result<RbJson, serde_json::error::Error> where T: Serialize {
        let j = serde_json::to_value(v)?;
        return Ok(RbJson {
            rb_json: j
        });
    }
}

impl From<serde_json::Value> for RbJson {
    fn from(arg: Value) -> Self {
        Self {
            rb_json: arg
        }
    }
}

impl<'de> serde::Deserialize<'de> for RbJson {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: serde::de::Deserializer<'de> {
        let mut js = serde_json::Value::deserialize(deserializer)?;
        if js.is_object() {
            if let Some(obj) = js.as_object_mut() {
                if obj.len() == 1 {
                    if let Some(v) = obj.remove("rb_json") {
                        return Ok(RbJson {
                            rb_json: v,
                        });
                    }
                }
            }
        }
        return Ok(RbJson {
            rb_json: js,
        });
    }
}

#[cfg(test)]
mod test {
    use crate::types::json::RbJson;

    #[test]
    fn test_decode() {
        let v: RbJson = serde_json::from_str("{\"rb_json\":1}").unwrap();
        println!("{:?}", v);
        let v: RbJson = serde_json::from_str("{\"ds\":1}").unwrap();
        println!("{:?}", v);
        let v: RbJson = serde_json::from_str("1").unwrap();
        println!("{:?}", v);
        let v: RbJson = serde_json::from_str("\"string\"").unwrap();
        println!("{:?}", v);
        let v: RbJson = serde_json::from_str("true").unwrap();
        println!("{:?}", v);
    }
}