use serde::{Serialize, Deserialize, Serializer, Deserializer};
use std::convert::TryFrom;
use serde::de::{Error, Visitor};
use std::fmt;
use std::ops::{Deref, DerefMut};
use serde_json::Value;

/// bytes
#[derive(Eq, PartialEq, Clone, Debug, serde::Serialize)]
pub struct RbBytes {
    pub rb_bytes: Vec<u8>,
}

impl RbBytes {
    pub fn new() -> Self {
        Self {
            rb_bytes: Vec::new()
        }
    }
}

impl From<serde_json::Value> for RbBytes {
    #[inline]
    fn from(arg: Value) -> Self {
        return match arg {
            Value::Array(arr) => {
                let mut bytes = Vec::with_capacity(arr.len());
                for x in arr {
                    bytes.push(x.as_u64().unwrap_or_default() as u8)
                }
                Self {
                    rb_bytes: bytes
                }
            }
            _ => {
                Self {
                    rb_bytes: vec![]
                }
            }
        };
    }
}

impl Deref for RbBytes {
    type Target = Vec<u8>;

    fn deref(&self) -> &Self::Target {
        &self.rb_bytes
    }
}

impl DerefMut for RbBytes {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.rb_bytes
    }
}


impl<'de> serde::Deserialize<'de> for RbBytes {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: serde::de::Deserializer<'de> {
        let mut js = serde_json::Value::deserialize(deserializer)?;
        if js.is_object() {
            if let Some(m) = js.as_object_mut() {
                if m.len() == 1 {
                    if let Some(v) = m.remove("rb_bytes") {
                        let bytes = serde_json::from_value(v);
                        if bytes.is_err() {
                            return Err(D::Error::custom(bytes.err().unwrap()));
                        }
                        return Ok(RbBytes {
                            rb_bytes: bytes.unwrap(),
                        });
                    }
                }
            }
        } else if js.is_array() {
            let js = serde_json::from_value(js);
            if js.is_err() {
                return Err(D::Error::custom(js.err().unwrap()));
            }
            return Ok(RbBytes {
                rb_bytes: js.unwrap(),
            });
        }
        return Err(D::Error::custom("unsupported decode json type!"));
    }
}


#[cfg(test)]
mod test {
    use crate::types::byte::RbBytes;

    #[test]
    pub fn test_encode() {
        let b = "ssss".as_bytes().to_owned();
        let v = serde_json::to_string(&b).unwrap_or_default();
        println!("{}", v);
    }

    #[test]
    fn test_decode() {
        let v: RbBytes = serde_json::from_str("{\"rb_bytes\":[1,2]}").unwrap();
        println!("{:?}", v);
        let v: RbBytes = serde_json::from_str("[1,2]").unwrap();
        println!("{:?}", v);
    }
}