/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate sdl2;

use std::rc::Rc;

use sdl2::{
    pixels::Color,
    ttf::Sdl2TtfContext,
    ttf::Font as sdl_Font
};

use crate::base::{
    BASE_DIR,
    ARGB,
    RGB
};

use super::{
    image::Image,
    System
};

pub struct FontContext {
    sdl_context: Sdl2TtfContext
}

pub struct Font<'a> {
    sdl_font: sdl_Font<'a, 'static>
}

impl FontContext {
    pub fn init() -> Result<FontContext, String> {
        let sdl_context = sdl2::ttf::init().map_err(|e| e.to_string())?;
        Ok(FontContext{sdl_context})
    }
}

impl<'a> Font<'a> {
    pub fn height(&self) -> i32 {
        self.sdl_font.height()
    }
}

impl System {
    pub fn load_font<'a>(ctx: &'a Rc<FontContext>, filepath: impl ToString, size: i32) -> Result<Font<'a>, String> {
        let filepath = filepath.to_string();

        match ctx.sdl_context.load_font(format!("{}/{}", BASE_DIR, filepath), size as u16) { // why not self.sdl_context.load_font()? - because of lifetimes...
            Ok(sdl_font) => Ok(Font{sdl_font}),
            Err(s) => Err(s)
        }
    }

    pub fn make_text_image(font: &Font, text: impl ToString, wrap: i32, color: RGB) -> Result<Image, String> {
        let text = text.to_string();

        let surface = if wrap > 0 {
            font.sdl_font.render(&text).blended_wrapped(Color::RGB(color.r, color.g, color.b), wrap as u32)
        } else {
            font.sdl_font.render(&text).blended(Color::RGB(color.r, color.g, color.b))
        };
        match surface {
            Ok(surface) => {
                match surface.without_lock() {
                    Some(raw) => {
                        // There's pitch... will it always be width * 4?
                        Ok(Image::make(surface.width() as i32, surface.height() as i32, raw.to_vec()))
                    },
                    None => Err(format!("sdl surface requires locking"))
                }
            },
            Err(e) => Err(e.to_string())
        }
    }

    pub fn font_context_rclone(&self) -> Rc<FontContext> {
        Rc::clone(&(self.font_context))
    }

    pub fn draw_text_image(&mut self, txt_img: &Image, shade: u8, margin: i32, x: i32, y: i32) {
        let _ = self.draw_rect(x, y, x + txt_img.width() - 1 + (margin << 1), y + txt_img.height() - 1 + (margin << 1), ARGB{a: shade, r: 0, g: 0, b: 0}, true);
        self.draw_image(txt_img, x + margin, y + margin);
    }

    /// Slow to do it every frame.
    /// Prepare image with make_text_image() once and draw it with draw_text_image() every frame...
    pub fn draw_text(&mut self, font: &Font, text: impl ToString, wrap: i32, color: RGB, shade: u8, margin: i32, x: i32, y: i32) {
        let text = text.to_string();

        if let Ok(txt_img) = Self::make_text_image(font, text, wrap, color) {
            self.draw_text_image(&txt_img, shade, margin, x, y)
        }
    }
}
