/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate image;
extern crate rayon;

use image::Pixel;
use rayon::prelude::*;

use crate::base::{
	BASE_DIR,
	RGB
};

use super::System;

#[derive(Clone)]
pub struct Image{
    width: i32,
    height: i32,
    buffer: Vec<u8>
}

impl Image {
    pub fn make(width: i32, height: i32, buffer: Vec<u8>) -> Self {
        Self{width, height, buffer}
    }

	pub fn load(base_filepath: impl ToString) -> Result<Self, String> {
		let base_filepath = base_filepath.to_string();

		match image::open(format!("{}/{}", BASE_DIR, base_filepath)) {
			Ok(img_buf) => {
				let img_buf = img_buf.into_bgra();

				let width: i32 = img_buf.dimensions().0 as i32;
				let height: i32 = img_buf.dimensions().1 as i32;
				let mut buffer: Vec<u8> = vec![0; ((width as usize) * (height as usize)) << 2];

				let mut offs: usize = 0;
				for y in 0..height {
					for x in 0..width {
						let pixel = img_buf.get_pixel(x as u32, y as u32).to_bgra();
						for c in 0..4 {
							buffer[offs + c] = pixel[c];
						}
						offs += 4;
					}
				}

				Ok(Self{width, height, buffer})
			},
			Err(e) => Err(e.to_string())
		}
	}


    pub fn width(&self) -> i32 {
        self.width
    }

    pub fn height(&self) -> i32 {
        self.height
    }

    #[inline]
    pub fn pixel(&self, x: i32, y: i32) -> Option<[u8; 4]> {
        if (x >= 0) && (x < self.width) && (y >= 0) && (y < self.height) {
            let buf = &self.buffer;
            let offs = ((y * self.width + x) << 2) as usize;
            Some([buf[offs+0], buf[offs+1], buf[offs+2], buf[offs+3]])
        } else {
            None
        }
    }

    #[inline(always)]
    pub fn pixel_nocheck(&self, x: i32, y: i32) -> [u8; 4] {
        let buf = &self.buffer;
        let offs = ((y * self.width + x) << 2) as usize;
        [buf[offs+0], buf[offs+1], buf[offs+2], buf[offs+3]]
    }

    pub fn raw(&self) -> &[u8] {
        &self.buffer
    }

    pub fn quarterized(&self) -> Image {
		let half_width: i32 = self.width >> 1;
		let half_height: i32 = self.height >> 1;

		let mut quart_buf: Vec<u8> = vec![0; ((half_width as usize) * (half_height as usize)) << 2];
		let buf = & self.buffer;

		let mut offs: usize = 0;
		let mut offs00: usize = 0;
		let mut offs10: usize = 4;
		let mut offs01: usize = (self.width << 2) as usize;
		let mut offs11: usize = ((self.width << 2) + 4) as usize;

		for _y in 0..half_height {
			for _x in 0..half_width {
				for c in 0..4 {
					quart_buf[offs + c] = (((buf[offs00 + c] as u32) + (buf[offs10 + c] as u32) + (buf[offs01 + c] as u32) + (buf[offs11 + c] as u32)) >> 2) as u8;
				}

				offs += 4;
				offs00 += 8; offs10 += 8; offs01 += 8; offs11 += 8;
			}
			offs00 += (self.width << 2) as usize; offs10 += (self.width << 2) as usize; offs01 += (self.width << 2) as usize; offs11 += (self.width << 2) as usize;
		}

		Image{width: half_width, height: half_height, buffer: quart_buf}
    }

	pub fn scaled(&self, scale: f64) -> Image {
		let scale = scale.abs();

		let width = ((self.width as f64) * scale) as i32;
		let height = ((self.height as f64) * scale) as i32;
		let mut buffer: Vec<u8> = vec![0; ((width as usize) * (height as usize)) << 2];

		if (width > 0) && (height > 0) {
			let inv_scale = 1.0 / scale;

			let mut offs: usize = 0;
			let mut oy = 0.0f64;
			for _y in 0..height {
				let y0 = (oy.floor() as i32).max(0).min(self.height - 1);
				let y1 = (y0 + 1).max(0).min(self.height - 1);
				let v = ((255.0 * (oy - oy.floor())) as i32) & 0xFF;
				let nv = 0xFF - v;

				let mut ox = 0.0f64;
				for _x in 0..width {
					let x0 = (ox.floor() as i32).max(0).min(self.width - 1);
					let x1 = (x0 + 1).max(0).min(self.width - 1);
					let u = ((255.0 * (ox - ox.floor())) as i32) & 0xFF;
					let nu = 0xFF - u;

					let c00 = self.pixel_nocheck(x0, y0);
					let c10 = self.pixel_nocheck(x1, y0);
					let c01 = self.pixel_nocheck(x0, y1);
					let c11 = self.pixel_nocheck(x1, y1);

					for i in 0..4 {
						buffer[offs + i] = (( ((c00[i] as i32) * nu + (c10[i] as i32) * u) * nv + ((c01[i] as i32) * nu + (c11[i] as i32) * u) * v ) >> 0x10) as u8;
					}

					ox += inv_scale;
					offs += 4;
				}

				oy += inv_scale;
			}
		}

		Image{width, height, buffer}
	}

	/// Make opaque with color as background ("draw on background filled with color")
	pub fn alphacolor(&mut self, color: RGB) {
		let color = [color.b as u32, color.g as u32, color.r as u32];
		let n = self.width * self.height;
		let mut offs: usize = 0;
		let buf = &mut self.buffer;
		for _i in 0..n {
			let a = buf[offs + 3] as u32;
			let na = 0x100 - a;
			for i in 0..3 {
				buf[offs + i] = (((buf[offs + i] as u32) * a + color[i] * na) >> 8) as u8;
			}
			buf[offs + 3] = 0xFF;
			offs += 4;
		}
	}

}

impl System {
	pub fn draw_image(&mut self, img: &Image, x: i32, y: i32) {
		let (width, height, scr_width, scr_height) = (img.width, img.height, self.width, self.height);
		if (x < scr_width) && (x + width > 0) && (y < scr_height) && (y + height > 0) { // intersects with screen
			// Corners of intersection in image's RF
			let (u0, v0) = (0i32.max(-x), 0i32.max(-y));
			let (u1, v1) = (width.min(scr_width - x), height.min(scr_height - y));

			let (y0, y1) = ((y + v0) as usize, (y + v1) as usize);

			let i2v_diff = v0 - (y0 as i32);

			let img_buf = &img.buffer;

			let row_offs_0 = ((x + u0) << 2) as usize;

			// Split into non-overlapping chunks (rows)
			let mut scr_rows: Vec<(usize, &mut [u8])> = self.buffer.chunks_mut((scr_width << 2) as usize)
				.enumerate()
				.collect();
			let scr_sub_rows = &mut scr_rows[y0..y1]; // rows intersecting with image
			// Parallelize by rayon iterator
			scr_sub_rows.into_par_iter().for_each(|(i, vb_row)| {
				let v = (*i as i32) + i2v_diff;
				let mut img_offs = ((v * width + u0) << 2) as usize;
				let mut row_offs = row_offs_0;
				for _u in u0..u1 {
					let a = img_buf[img_offs + 3] as u32;
					let na = 0x100 - a;
					for i in 0..3 { // Compiler should unroll this...
						vb_row[row_offs + i] = (((img_buf[img_offs + i] as u32) * a + (vb_row[row_offs + i] as u32) * na) >> 8) as u8;
					}
					img_offs += 4;
					row_offs += 4;
				}
			});
		}
	}

}
