/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate rayon;

use rayon::prelude::*;

use crate::base::ARGB;

use super::System;

impl System {
    /// Draw line
    /// Bresenham's, from Rosetta Code
    #[inline]
    pub fn draw_line(&mut self, mut x1: i32, mut y1: i32, x2: i32, y2: i32, color: ARGB) -> Result<(), String> {
        let (dx, sx) = ((x2 - x1).abs(), (x2 - x1).signum());
        let (dy, sy) = ((y2 - y1).abs(), (y2 - y1).signum());

        let mut err = dx - dy;
        let mut e2: i32;

        let a = color.a as u32;
        let na = 0x100 - a;

        let color_mult_a: [u32; 3] = [(color.b as u32) * a, (color.g as u32) * a, (color.r as u32) * a];

        let (w, h) = (self.width, self.height);
        let buf = &mut self.buffer;

        loop {
            if (x1 >= 0) && (y1 >= 0) && (x1 < w) && (y1 < h) {
                let offs = ((y1 * w + x1) << 2) as usize;
                for c in 0..3 {
                    buf[offs + c] = ((color_mult_a[c] + (buf[offs + c] as u32) * na) >> 8) as u8;
                }
            }

            if (x1 == x2) && (y1 == y2) {
                break;
            }

            e2 = err << 1;
            if e2 > -dy {
                err -= dy;
                x1 += sx;
            }
            if e2 < dx {
                err += dx;
                y1 += sy;
            }
        }

        Ok(())
    }

    #[inline]
    pub fn draw_rect(&mut self, x1: i32, y1: i32, x2: i32, y2: i32, color: ARGB, fill: bool) -> Result<(), String> {
        let (x1, x2, y1, y2) = (x1.min(x2), x1.max(x2), y1.min(y2), y1.max(y2));

        if fill {
            if (x1 < self.width) && (x2 >= 0) && (y1 < self.height) && (y2 >= 0) { // intersects with screen
                let a = color.a as u32;
                let na = 0x100 - a;

                let color_mult_a: [u32; 3] = [(color.b as u32) * a, (color.g as u32) * a, (color.r as u32) * a];

                let x1 = x1.max(0) as usize;
                let y1 = y1.max(0) as usize;

                let x2 = (x2 + 1).min(self.width) as usize;
                let y2 = (y2 + 1).min(self.height) as usize;

                let offs_0 = x1 << 2;

                let mut vb_rows: Vec<&mut [u8]> = self.buffer.chunks_mut((self.width << 2) as usize).collect();
                let vb_sub_rows = &mut vb_rows[y1..y2]; // screen rows intersecting with rectangle
                // Parallelize by rayon iterator
                vb_sub_rows.into_par_iter().for_each(|vb_row| {
                    let mut offs = offs_0;
                    for _x in x1..x2 {
                        for i in 0..3 {
                            vb_row[offs + i] = ((color_mult_a[i] + (vb_row[offs + i] as u32) * na) >> 8) as u8;
                        }
                        offs += 4;
                    }
                });
            }
        } else {
            // FIX: Corner pixels are drawn twice, alpha accumulates
            let _ = self.draw_line(x1, y1, x2, y1, color);
            let _ = self.draw_line(x2, y1, x2, y2, color);
            let _ = self.draw_line(x2, y2, x1, y2, color);
            let _ = self.draw_line(x1, y2, x1, y1, color);
        }

        Ok(())
    }

    #[inline]
    pub fn shade(&mut self, shade: u8) -> Result<(), String> {
        self.draw_rect(0, 0, self.width - 1, self.height - 1, ARGB{a: shade, r: 0, g: 0, b: 0}, true)
    }

    #[inline]
    pub fn draw_frame(&mut self,
        x: i32, y: i32, width: i32, height: i32, // interior rectangle
        color_int: ARGB, color_ext: ARGB, // internal and external edge color
        thickness_binlog: i32) -> Result<(), String> {

        let th: i32 = 1 << thickness_binlog;
        for i in 1..=th {
            let ni = th - i;
            let color = ARGB {
                a: (((color_ext.a as i32) * i + (color_int.a as i32) * ni) >> thickness_binlog) as u8,
                r: (((color_ext.r as i32) * i + (color_int.r as i32) * ni) >> thickness_binlog) as u8,
                g: (((color_ext.g as i32) * i + (color_int.g as i32) * ni) >> thickness_binlog) as u8,
                b: (((color_ext.b as i32) * i + (color_int.b as i32) * ni) >> thickness_binlog) as u8
            };
            self.draw_rect(x-i, y-i, x+width+i-1, y+height+i-1, color, false)?;
        }

        Ok(())
    }

}
