/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate rayon;
extern crate sdl2;

use core::slice::ChunksMut;

use std::rc::Rc;

use rayon::prelude::*;

use sdl2::{
	pixels::Color,
	pixels::PixelFormatEnum,
	render::Texture,
	render::WindowCanvas
};

use crate::base::PrependErrorString;

use super::WindowMode;

mod geom;
mod image;
mod pixel;
mod text;

pub use self::{
	image::Image,
	text::Font,
	text::FontContext
};

pub struct System {
    width: i32,
    height: i32,

    buffer: Vec<u8>,
	tetra_buffer: Vec<u8>,

    sdl_screen_canvas: WindowCanvas,
	sdl_screen_texture: Texture, // requires "features = [..., "unsafe_textures"]" under [dependencies.sdl2] in Cargo.toml, to avoid lifetimes

	font_context: Rc<FontContext>
}

impl System {
    pub fn init(sdl: &sdl2::Sdl, window_name: String, window_mode: WindowMode, mut width: i32, mut height: i32, vsync: bool) -> Result<System, String> {
        let video_subsystem = sdl.video().pre_err("cannot init sdl video")?;
        video_subsystem.disable_screen_saver();

		if window_mode == WindowMode::FullscreenDesktop {
			width = video_subsystem.current_display_mode(0)?.w;
			height = video_subsystem.current_display_mode(0)?.h;
		}

		if (width & 1) != 0 {
            return Err(format!("screen width {} is not even", width));
        }

        if (height & 1) != 0 {
            return Err(format!("screen height {} is not even", height));
        }

        let buffer: Vec<u8> = vec![0xFF; ((width as usize) * (height as usize)) << 2];
		let tetra_buffer: Vec<u8> = vec![0xFF; (((width << 1) as usize) * ((height << 1) as usize)) << 2];

		let mut window_builder = video_subsystem.window(window_name.as_str(), width as u32, height as u32);
		match window_mode {
			WindowMode::Windowed => {window_builder.position_centered();},
			WindowMode::Fullscreen => {window_builder.fullscreen();},
			WindowMode::FullscreenDesktop => {window_builder.fullscreen_desktop();}
		}

        let window = window_builder.build().pre_err("cannot build sdl window")?;

		let mut canvas_builder = window.into_canvas();
		if vsync {
			canvas_builder = canvas_builder.present_vsync();
		}
        let mut sdl_screen_canvas = canvas_builder.build().pre_err("cannot build sdl canvas")?;

        sdl_screen_canvas.set_draw_color(Color::RGB(0, 0, 0));
        sdl_screen_canvas.clear();
        sdl_screen_canvas.present();

		let sdl_texture_creator = sdl_screen_canvas.texture_creator();

		let sdl_screen_texture = sdl_texture_creator.create_texture_streaming(Some(PixelFormatEnum::ARGB8888), width as u32, height as u32)
			.pre_err("cannot create sdl texture")?;

		let font_context = Rc::new(FontContext::init().pre_err("cannot init sdl font context")?);

        Ok(System{width, height, buffer, tetra_buffer, sdl_screen_canvas, sdl_screen_texture, font_context})
    }

	#[inline]
	pub fn width(&self) -> i32 {
		self.width
	}

	#[inline]
	pub fn height(&self) -> i32 {
		self.height
	}

	// Display video buffer in window
	pub fn show(&mut self) -> Result<(), String> {
		self.sdl_screen_texture.update(None, &self.buffer, (self.width as usize) << 2)
			.pre_err("cannot update sdl texture from buffer")?;

		// self.sdl_screen_canvas.clear();
		self.sdl_screen_canvas.copy(&self.sdl_screen_texture, None, None).pre_err("cannot copy sdl texture to sdl canvas")?;
		self.sdl_screen_canvas.present();

		Ok(())
	}

	pub fn screen_raw_mut(&mut self) -> &mut [u8] {
		&mut self.buffer
	}

	pub fn screen_raw_chunks_mut(&mut self, pixels_per_chunk: i32) -> ChunksMut<u8> {
		self.buffer.chunks_mut((pixels_per_chunk << 2) as usize)
	}

	pub fn tetra_screen_raw_mut(&mut self) -> &mut [u8] {
		&mut self.tetra_buffer
	}

	pub fn tetra_screen_raw_chunks_mut(&mut self, pixels_per_chunk: i32) -> ChunksMut<u8> {
		self.tetra_buffer.chunks_mut((pixels_per_chunk << 2) as usize)
	}

	pub fn quarterize(&mut self) {
		let width = self.width as usize;
		let tetra_buf = & self.tetra_buffer;

		let rows: Vec<(usize, &mut [u8])> = self.buffer.chunks_mut((self.width << 2) as usize).enumerate().collect();
		// Parallelize by rayon iterator
		rows.into_par_iter()
			.for_each(|(y, row)| {
				let mut offs = 0;

				let mut offs00 = (y * width) << 4;
				let mut offs10 = offs00 + 4;
				let mut offs01 = offs00 + (width << 3);
				let mut offs11 = offs01 + 4;

				for _x in 0..width {
					for c in 0..3 {
						row[offs + c] = (((tetra_buf[offs00 + c] as u32) + (tetra_buf[offs10 + c] as u32) + (tetra_buf[offs01 + c] as u32) + (tetra_buf[offs11 + c] as u32)) >> 2) as u8;
					}

					offs += 4;
					offs00 += 8; offs10 += 8; offs01 += 8; offs11 += 8;
				}
			});
	}

	pub fn shut(&mut self) -> Result<(), String> {
		Ok(())
	}

}
