/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate sdl2;

use core::slice::ChunksMut;

use std::rc::Rc;

use crate::base::{
	ARGB,
	RGB
};

use super::System;

use super::video::{
    self,
    Font,
    FontContext,
    Image
};

const ERR_DRAW_LOCKED: &str = "draw locked";

impl System {
    pub fn load_font<'a>(ctx: &'a Rc<FontContext>, filepath: impl ToString, size: i32) -> Result<Font<'a>, String> {
		let filepath = filepath.to_string();
        video::System::load_font(ctx, filepath, size)
    }

	pub fn make_text_image(font: &Font, text: impl ToString, wrap: i32, color: RGB) -> Result<Image, String> {
		let text = text.to_string();
        video::System::make_text_image(font, text, wrap, color)
    }

    #[inline]
    pub fn draw_pixel(&mut self, x: i32, y: i32, color: ARGB) -> Result<(), String> {
        if !self.draw_locked {
			let (x, y) = (x + (self.width() >> 1), y + (self.height() >> 1));
            self.video.draw_pixel(x, y, color)
        } else {
            Err(String::from(ERR_DRAW_LOCKED))
        }
    }

    #[inline]
    pub fn draw_line(&mut self, x1: i32, y1: i32, x2: i32, y2: i32, color: ARGB) -> Result<(), String> {
        if !self.draw_locked {
			let (x1, y1) = (x1 + (self.width() >> 1), y1 + (self.height() >> 1));
			let (x2, y2) = (x2 + (self.width() >> 1), y2 + (self.height() >> 1));
            self.video.draw_line(x1, y1, x2, y2, color)
        } else {
            Err(String::from(ERR_DRAW_LOCKED))
        }
    }

    #[inline]
    pub fn draw_rect(&mut self, x1: i32, y1: i32, x2: i32, y2: i32, color: ARGB, fill: bool) -> Result<(), String> {
        if !self.draw_locked {
			let (x1, y1) = (x1 + (self.width() >> 1), y1 + (self.height() >> 1));
			let (x2, y2) = (x2 + (self.width() >> 1), y2 + (self.height() >> 1));
            self.video.draw_rect(x1, y1, x2, y2, color, fill)
        } else {
            Err(String::from(ERR_DRAW_LOCKED))
        }
    }

	#[inline]
	pub fn shade(&mut self, shade: u8) -> Result<(), String> {
		if !self.draw_locked {
			self.video.shade(shade)
		} else {
			Err(String::from(ERR_DRAW_LOCKED))
		}
	}

	#[inline]
	pub fn draw_frame(&mut self,
		x: i32, y: i32, width: i32, height: i32, // interior rectangle
		color_int: ARGB, color_ext: ARGB, // internal and external edge color
		thickness_binlog: i32) -> Result<(), String> {
		if !self.draw_locked {
			let (x, y) = (x + (self.width() >> 1), y + (self.height() >> 1));
			self.video.draw_frame(x, y, width, height, color_int, color_ext, thickness_binlog)
		} else {
			Err(String::from(ERR_DRAW_LOCKED))
		}
	}

    pub fn draw_image(&mut self, img: &Image, x: i32, y: i32) -> Result<(), String> {
        if !self.draw_locked {
			let (x, y) = (x + (self.width() >> 1), y + (self.height() >> 1));
            self.video.draw_image(img, x, y);
			Ok(())
        } else {
            Err(String::from(ERR_DRAW_LOCKED))
        }
    }

    pub fn font_context_rclone(&self) -> Rc<FontContext> {
        self.video.font_context_rclone()
    }

    pub fn draw_text_image(&mut self, txt_img: &Image, shade: u8, margin: i32, x: i32, y: i32) -> Result<(), String> {
        if !self.draw_locked {
			let (x, y) = (x + (self.width() >> 1), y + (self.height() >> 1));
            self.video.draw_text_image(txt_img, shade, margin, x, y);
			Ok(())
        } else {
            Err(String::from(ERR_DRAW_LOCKED))
        }
    }

    /// Slow to do it every frame.
    /// Prepare image with make_text_image() once and draw it with draw_text_image() every frame...
    pub fn draw_text(&mut self, font: &Font, text: impl ToString, wrap: i32, color: RGB, shade: u8, margin: i32, x: i32, y: i32) -> Result<(), String> {
        if !self.draw_locked {
			let text = text.to_string();
			let (x, y) = (x + (self.width() >> 1), y + (self.height() >> 1));
            self.video.draw_text(font, text, wrap, color, shade, margin, x, y);
			Ok(())
        } else {
            Err(String::from(ERR_DRAW_LOCKED))
        }
    }

	pub fn screen_raw_mut(&mut self) -> &mut [u8] {
		match self.antialiasing_4x() {
			true => self.video.tetra_screen_raw_mut(),
			false => self.video.screen_raw_mut()
		}
	}

	pub fn screen_raw_chunks_mut(&mut self, pixels_per_chunk: i32) -> ChunksMut<u8> {
		match self.antialiasing_4x() {
			true => self.video.tetra_screen_raw_chunks_mut(pixels_per_chunk),
			false => self.video.screen_raw_chunks_mut(pixels_per_chunk)
		}
	}

	pub fn quarterize(&mut self) {
		if (!self.draw_locked) && self.config.antialiasing_4x() {
			self.video.quarterize();
		}
	}

    pub fn draw_locked(&self) -> bool {
        self.draw_locked
    }

}
