/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate sdl2;

use sdl2::mixer::{
    self,
    Channel,
    Chunk,
    Music as sdl_Music
};

use crate::base::BASE_DIR;

const FREQUENCY: i32 = 44100;
const FORMAT_DATA: mixer::AudioFormat = mixer::DEFAULT_FORMAT;
const FORMAT_CHANNELS: i32 = 2;
const CHUNKSIZE: i32 = 1024;

const CHANNELS: i32 = 0x10;

pub const MUSICS_DIR: &str = "audio/musics";
pub const SOUNDS_DIR: &str = "audio/sounds";
pub const SPEECHES_SUBDIR: &str = "speeches";

const MUSIC_FADE_OUT_DELAY: i32 = 1000; // milsec
const MUSIC_FADE_IN_DELAY: i32 = 1000; // milsec

const SPEECH_FADE_OUT_DELAY: i32 = 200; // milsec
const SPEECH_FADE_IN_DELAY: i32 = 200; // milsec

pub struct System {
    mute: bool,
    vol_sound: i32,
    vol_music: i32,
    vol_speech: i32
}

pub struct Music<'a> {
    sdl_music: sdl_Music<'a>
}

pub struct Sound {
    chunk: Chunk
}

impl System {
    pub fn init(mute: bool, vol_sound: i32, vol_music: i32, vol_speech: i32) -> Result<System, String> {
        if !mute {
            mixer::open_audio(FREQUENCY, FORMAT_DATA, FORMAT_CHANNELS, CHUNKSIZE)?; // here "channels" are 1 (mono) / 2 (stereo), not channels to play chunks through...
            mixer::allocate_channels(CHANNELS);
            sdl_Music::set_volume(vol_music * 128 / 100);
            mixer::reserve_channels(1); // 0-th channel to play speech chunk through
        }

        Ok(System{mute, vol_sound, vol_music, vol_speech})
    }

    pub fn load_music<'a>(&self, filepath: impl ToString) -> Option<Music<'a>> {
        if !self.mute {
            let filepath = filepath.to_string();
            match sdl_Music::from_file(format!("{}/{}/{}", BASE_DIR, MUSICS_DIR, filepath)) {
                Ok(sdl_music) => Some(Music{sdl_music}),
                Err(..) => None
            }
        } else {
            None
        }
    }

    pub fn play_music(&self, music: Option<&Music>) -> Result<(), String> {
        if !self.mute {
            match music {
                Some(m) => {
                    // sdl_Music::halt();
                    // sdl_Music::fade_out(MUSIC_FADE_OUT_DELAY)?; // not async, freezes main thread for some time...
                    // std::thread::spawn(|| {let _ = sdl_Music::fade_out(MUSIC_FADE_OUT_DELAY);} ); // silence, "new" music does not start... too asynced?

                    // m.sdl_music.play(-1)?;
                    m.sdl_music.fade_in(-1, MUSIC_FADE_IN_DELAY)?; // lessens (?) "click" between musics

                    Ok(())
                },
                None => Err(String::from("no music to play"))
            }
        } else {
            Ok(())
        }
    }

    fn load_chunk(&self, base_dirpath: impl ToString, filepath: impl ToString, volume: i32) -> Option<Sound> {
        if !self.mute {
            let base_dirpath = base_dirpath.to_string();
            let filepath = filepath.to_string();
            match Chunk::from_file(format!("{}/{}/{}", BASE_DIR, base_dirpath, filepath)) {
                Ok(mut chunk) => {
                    chunk.set_volume(volume * 128 / 100);
                    Some(Sound{chunk})
                },
                Err(..) => None
            }
        } else {
            None
        }
    }

    pub fn load_sound(&self, filepath: impl ToString) -> Option<Sound> {
        self.load_chunk(SOUNDS_DIR, filepath, self.vol_sound)
    }

    pub fn play_sound(&self, sound: Option<&Sound>, loops: i32) -> Result<(), String> {
        if !self.mute {
            match sound {
                Some(s) => {
                    sdl2::mixer::Channel::all().play(&s.chunk, loops)?;
                    Ok(())
                },
                None => Err(String::from("no sound to play"))
            }
        } else {
            Ok(())
        }
    }

    pub fn load_speech(&self, filepath: impl ToString) -> Option<Sound> {
        self.load_chunk(format!("{}/{}", SOUNDS_DIR, SPEECHES_SUBDIR), filepath, self.vol_speech)
    }

    pub fn play_speech(&self, speech: Option<&Sound>) -> Result<(), String> {
        if !self.mute {
            let chan = Channel(0); // reserved for speech
            // chan.halt(); // stop already playing speech if any
            // Fade out/in to reduce (?) "click" between speeches
            chan.fade_out(SPEECH_FADE_OUT_DELAY);
            match speech {
                Some(s) => {
                    // chan.play(&s.chunk, 0)?;
                    chan.fade_in(&s.chunk, 0, SPEECH_FADE_IN_DELAY)?;
                    Ok(())
                },
                None => Err(String::from("no speech to play"))
            }
        } else {
            Ok(())
        }
    }

    pub fn pause(&self) {
        if !self.mute {
            sdl_Music::pause();
            Channel::all().pause();
            Channel(0).pause();
        }
    }

    pub fn resume(&self) {
        if !self.mute {
            sdl_Music::resume();
            Channel::all().resume();
            Channel(0).resume();
        }
    }

    pub fn halt(&self) {
        if !self.mute {
            sdl_Music::halt();
            Channel::all().halt();
            Channel(0).halt();
        }
    }

    pub fn shut(&mut self) -> Result<(), String> {
        if !self.mute {
            mixer::close_audio();
        }
        Ok(())
    }
}
