/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

//! Mostly SDL wrappers. Also, load config, get time since start etc.

extern crate sdl2;
extern crate num_cpus;
extern crate serde;

use serde::Deserialize;

use crate::base::PrependErrorString;

mod audio;
mod config;
mod input;
mod random;
mod time;
mod video;

mod sys_audio;
mod sys_input;
mod sys_random;
mod sys_sync;
mod sys_time;
mod sys_video;

pub use self::audio::{
    SOUNDS_DIR,
    SPEECHES_SUBDIR,
    Music,
    Sound
};

pub use self::input::{
    Button,
    Key
};

pub use self::video::{
    Font,
    FontContext,
    Image,
};

#[derive(Clone, Copy, Deserialize, Eq, PartialEq)]
pub enum WindowMode {
    Windowed,
    Fullscreen,
    FullscreenDesktop
}

#[derive(Clone, Copy, Deserialize, Eq, PartialEq)]
pub enum CpuCountMode {
    Logical,
    Physical
}

pub struct System {
    config: Box<self::config::Config>,
    num_cpus: i32,

    time: Box<self::time::System>,
    random: Box<self::random::System>,

    sdl: Box<sdl2::Sdl>,

    video: Box<self::video::System>,
    draw_locked: bool,
    last_show_time: i128,
    fps: Vec<f64>,
    fps_i: usize,

    audio: Box<self::audio::System>,

    input: Box<self::input::System>
}

impl System {
    pub fn init() -> Result<System, String> {
        let mut config = config::Config::load().pre_err("cannot load config")?;
        let num_cpus = match config.cpu_count_mode() {
            CpuCountMode::Logical => num_cpus::get(),
            CpuCountMode::Physical => num_cpus::get_physical()
        } as i32;

        let time = time::System::init().pre_err("cannot init time subsys")?;

        let random = random::System::init().pre_err("cannot init random subsys")?;

        let sdl = sdl2::init().pre_err("cannot init SDL")?;

        // Video first, it can change resolution in case of Fullscreen Desktop mode
        let video = video::System::init(&sdl, config.window_name(), config.window_mode(), config.width(), config.height(), config.vsync()).pre_err("cannot init video subsys")?;
        config.set_width(video.width());
        config.set_height(video.height());

        let draw_locked: bool = false;
        let last_show_time = time.now();
        let fps: Vec<f64> = vec![0.0; config.frame_rate() as usize];
        let fps_i: usize = 0;

        let audio = audio::System::init(config.mute(), config.vol_sound(), config.vol_music(), config.vol_speech()).pre_err("cannot init audio subsys")?;

        let input = input::System::init(&sdl, config.mouse_sensitivity(), config.left_hand()).pre_err("cannot init input subsys")?;

        Ok(System{
            config: Box::new(config),
            num_cpus,
            time: Box::new(time),
            random: Box::new(random),
            sdl: Box::new(sdl),
            video: Box::new(video),
            draw_locked,
            last_show_time,
            fps,
            fps_i,
            audio: Box::new(audio),
            input: Box::new(input)
        })
    }

    #[inline]
    pub fn width(&self) -> i32 {
        self.config.width()
    }

    #[inline]
    pub fn height(&self) -> i32 {
        self.config.height()
    }

    pub fn field_of_view(&self) -> i32 {
        self.config.field_of_view()
    }

    pub fn antialiasing_4x(&self) -> bool {
        self.config.antialiasing_4x()
    }

    pub fn texture_smooth(&self) -> bool {
        self.config.texture_smooth()
    }

    pub fn lang(&self) -> String {
        self.config.lang()
    }

    #[inline]
    pub fn num_cpus(&self) -> i32 {
        self.num_cpus
    }

    pub fn fps(&self) -> f64 {
        self.fps.iter().sum::<f64>() / (self.config.frame_rate() as f64)
    }

    pub fn shut(&mut self) -> Result<(), String> {
        self.input.shut().pre_err("cannot shut input subsys")?;
        self.audio.shut().pre_err("cannot shut audio subsys")?;
        self.video.shut().pre_err("cannot shut video subsys")?;
        self.random.shut().pre_err("cannot shut random subsys")?;
        self.time.shut().pre_err("cannot shut time subsys")?;
        Ok(())
    }

}
