/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

//! State of play

extern crate miniz_oxide;
extern crate serde;
extern crate serde_json;

use std::{
    fs,
    fs::File
};

use miniz_oxide::{
    deflate,
    deflate::CompressionLevel,
    inflate
};

use serde::{
    Deserialize,
    Serialize
};

use crate::base::{
    SAVES_DIR,
    Attitude,
    PrependErrorString
};

const INF_MAX_LOOK_DIST_BINLOG: i32 = 0x10 + 10;
const SUP_MAX_LOOK_DIST_BINLOG: i32 = 0x10 + 32;

pub const AUTOSAVE_FILEPATH: &str = "autosave.jzn";

// DEBUG
// Set to true (for save only, then start-resave-quit, then for both) when no new fields are added to State, e.g. at release
const DEFLATE_SAVE_JSON: bool = false;
const INFLATE_LOAD_JSON: bool = false;

#[derive(Clone, Copy, Hash, Eq, PartialEq, Debug, Serialize, Deserialize)]
pub enum Verse {
    Dawn,
    Dusk
}

impl Default for Verse {
    fn default() -> Self {
        Self::Dawn
    }
}

/// Save/load of it must be possible...
/// Cosmos may be affected by the State, but mostly the _appearance_ of the Cosmos to the player is affected by the State...
/// (Cosmos contains stage and dolls, State describes where the player and dolls must be placed, the mood of the dolls etc. Attitudes of dolls are in Cosmos though.)
/// "defaults" allow to add new fields without modifying all saves made before to avoid "missed field" deserialization error
#[derive(Serialize, Deserialize)]
pub struct State {
    #[serde(default)] pub verse: Verse,
    #[serde(default)] pub transversed: bool,
    #[serde(default)] pub next_verse: Verse,
    #[serde(default)] pub transpoint: String,
    #[serde(default)] pub prev_verse: Verse,

    #[serde(default)] pub plr_att: Attitude,
    #[serde(default)] pub plr_i_chamber: usize,

    #[serde(default = "def_items")] pub items: Vec<String>,
    #[serde(default)] pub sel_item: String,

    #[serde(default)] pub time: i128, // total play time in microseconds

    #[serde(default = "def_max_look_dist_binlog")] pub max_look_dist_binlog: i32,

    #[serde(default)] pub entered: bool,

    #[serde(default)] pub plate_taken: bool,

    #[serde(default)] pub yellow_helix_flew: bool,
    #[serde(default)] pub green_helix_talked: bool
}

fn def_items() -> Vec<String> {
    vec![String::from("memories"), String::from("melody")]
}

fn def_max_look_dist_binlog() -> i32 {
    20 + 0x10 // 0x10 due to FPA
}

impl State {
    pub fn new() -> State {
        State{
            verse: Default::default(),
            transversed: Default::default(),
            next_verse: Default::default(),
            transpoint: Default::default(),
            prev_verse: Default::default(),
            plr_att: Default::default(),
            plr_i_chamber: Default::default(),
            items: def_items(),
            sel_item: Default::default(),
            time: Default::default(),
            max_look_dist_binlog: def_max_look_dist_binlog(),

            entered: Default::default(),

            plate_taken: Default::default(),

            yellow_helix_flew: Default::default(),
            green_helix_talked: Default::default()
        }
    }

    pub fn savesdir_exists_or_made() -> bool {
        fs::create_dir_all(SAVES_DIR).is_ok()
    }

    pub fn savefile_exists(filepath: impl ToString) -> bool {
        let save_filepath = format!("{}/{}", SAVES_DIR, filepath.to_string());
        match File::open(&save_filepath) {
            Ok(_) => true,
            Err(_) => false
        }
    }

    pub fn load(filepath: impl ToString) -> Result<Self, String> {
        let save_filepath = format!("{}/{}", SAVES_DIR, filepath.to_string());
        let mut json = fs::read(&save_filepath).pre_err(format!("cannot read from '{}'", &save_filepath))?;
        if INFLATE_LOAD_JSON {
            json = inflate::decompress_to_vec(&json).map_err(|e| format!("cannot inflate '{}': error {}", &save_filepath, e as i8))?;
        }
        let state: Self = serde_json::from_slice(&json).pre_err(format!("cannot deserialize (inflated) '{}' to state", &save_filepath))?;
        Ok(state)
    }

    pub fn save(&self, filepath: impl ToString) -> Result<(), String> {
        let mut json = match DEFLATE_SAVE_JSON {
            false => serde_json::to_vec_pretty(self).pre_err("cannot serialize state")?,
            true => serde_json::to_vec(self).pre_err("cannot serialize state")?
        };
        if DEFLATE_SAVE_JSON {
            json = deflate::compress_to_vec(&json, CompressionLevel::BestCompression as u8);
        }
        let save_filepath = format!("{}/{}", SAVES_DIR, filepath.to_string());
        fs::write(&save_filepath, &json).pre_err(format!("cannot write to '{}'", &save_filepath))?; // no buffering, write all at once
        Ok(())
    }

    pub fn transverse(&mut self, verse: Verse, transpoint: impl ToString) {
        self.transversed = true;
        self.next_verse = verse;
        self.transpoint = transpoint.to_string();
    }

    pub fn item_ind(&self, item: impl ToString) -> Option<usize> {
        let item = item.to_string();
        if item.is_empty() {
            return None;
        }
        for i in 0..self.items.len() {
            if item.eq(&(self.items[i])) {
                return Some(i);
            }
        }
        None
    }

    pub fn sel_item_ind(&self) -> Option<usize> {
        self.item_ind(&(self.sel_item))
    }

    pub fn add_item(&mut self, item: impl ToString) -> Result<(), String> {
        let item = item.to_string();
        if self.sel_item.eq(&item) {
            return Err(format!("cannot add item '{}': selected already", &item))
        }
        if self.items.contains(&item) {
            return Err(format!("cannot add item '{}': acquired already", &item))
        }
        self.items.push(item);
        Ok(())
    }

    pub fn add_and_sel_item(&mut self, item: impl ToString) -> Result<(), String> {
        let item = item.to_string();
        self.add_item(&item)?;
        self.sel_item = item;
        Ok(())
    }

    pub fn remove_item(&mut self, item: impl ToString) -> Result<(), String> {
        let item = item.to_string();
        if self.sel_item.eq(&item) {
            self.sel_item.clear();
        }
        for i in 0..self.items.len() {
            if self.items[i].eq(&item) {
                self.items.remove(i);
                return Ok(())
            }
        }
        Err(format!("item '{}' not found", &item))
    }

    pub fn max_look_dist_binlog(&self) -> i32 {
        self.max_look_dist_binlog
    }

    pub fn set_max_look_dist_binlog(&mut self, v: i32) {
        self.max_look_dist_binlog = v.max(INF_MAX_LOOK_DIST_BINLOG).min(SUP_MAX_LOOK_DIST_BINLOG);
    }

}
