/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use std::f64::consts::PI;

use crate::{
	master::{
		ERR_ALREADY_SET,
		ERR_NOT_SET_CANNOT_RUN,
		Context,
		Master,
		Mode
	},
	sys::{
		Button,
		Key
	}
};

const MID: &str = "Helices";

const BLUE_HELIX_REACH_DIST: i64 = 20000 << 0x10;
const GREEN_HELIX_REACH_DIST: i64 = 30000 << 0x10;
const YELLOW_HELIX_REACH_DIST: i64 = 20000 << 0x10;

pub enum MHelices {
	Unset,
	Set {
		i_blue_hel: usize,
		i_green_hel: usize,
		i_yellow_hel: usize,

		i_quad_path: usize,
		i_helix_flight: usize,

		yellow_hel_flying: bool,
		yellow_hel_flying_start_time: i128
	}
}

impl MHelices {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}
}

impl Master for MHelices {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, Context{lingua, cosm, state, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				lingua.load(&["blue_helix", "green_helix", "yellow_helix"])?;
				let i_yellow_hel = cosm.entity_ind("yellow-helix")?;
				let i_helix_flight = cosm.trajectory_ind("1st helix flight")?;
				*self = Self::Set{
					i_blue_hel: cosm.entity_ind("blue-helix")?,
					i_green_hel: cosm.entity_ind("green-helix")?,
					i_yellow_hel,

					i_quad_path: cosm.trajectory_ind("quad-path-scaled")?,
					i_helix_flight,

					yellow_hel_flying: false,
					yellow_hel_flying_start_time: 0
				};

				let mut ent = &mut cosm.entities[i_yellow_hel];
				let traj = & cosm.trajectories[i_helix_flight];
				ent.att = match state.yellow_helix_flew {
					false => traj.first_attitude(),
					true => traj.last_attitude()
				};

				Ok(())
			},
			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, Context{sys, cosm, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Set{i_blue_hel, i_green_hel, i_yellow_hel, i_quad_path, i_helix_flight,
				ref mut yellow_hel_flying, ref mut yellow_hel_flying_start_time} => {

				let entities = &mut cosm.entities;

				// Blue helix
				let ent = &mut entities[i_blue_hel];
				for ball in &mut ent.balls {
					ball.color[0] = (200.0 + 55.0 * (0.75 * (state.time as f64) * 1e-6 * 2.0 * PI).sin() ) as u8;
				}
				if ent.visible {
					if ether.mode == Mode::Fly {
						if sys.poll_key(Key::V) {
							ent.visible = !ent.visible;
						}
						if sys.test_btn(Button::Middle) {
							ent.att.ori.yaw += (sys.mouse_dx() as f64) * 0.5 * PI / 180.0;
							ent.att.ori.pitch += (sys.mouse_dy() as f64) * 0.5 * PI / 180.0;
						}
					}

					if ether.sync_reach(&state.plr_att.pos, &ent.att.pos, BLUE_HELIX_REACH_DIST, ent.id()) {
						if (ether.mode == Mode::Fly) && ether.sel_reachable.eq(ent.id()) {
							if sys.poll_btn(Button::Right) {
								ether.describe("blue-helix");
							} else if sys.poll_btn(Button::Left) {
								match state.sel_item.as_str() {
									"" => { ent.att.pos.z += 1000 << 0x10; },
									"melody" => ether.describe("melody + blue-helix"),
									"memories" => ether.describe("memories + blue-helix"),
									"plate" => ether.describe("plate + blue-helix"),
									_ => ()
								};
							}
						}
					}
				}

				// Green helix
				let ent = &mut entities[i_green_hel];
				let traj = & cosm.trajectories[i_quad_path];
				ent.att = traj.attitude(state.time);
				if ent.visible && ether.sync_reach(&state.plr_att.pos, &ent.att.pos, GREEN_HELIX_REACH_DIST, ent.id()) {
					if (ether.mode == Mode::Fly) && ether.sel_reachable.eq(ent.id()) {
						if sys.poll_btn(Button::Right) {
							ether.describe("green-helix");
						} else if sys.poll_btn(Button::Left) {
							match state.sel_item.as_str() {
								"" => {
									match state.green_helix_talked {
										false => {
											state.green_helix_talked = true;
											ether.talk("green-helix", "greet 1st", &["greet polite", "greet impolite", "fear", "colors", "bye"]);
										},
										true => ether.talk("green-helix", "greet non-1st", &["greet polite", "greet impolite", "fear", "colors", "bye"])
									}
								},
								"plate" => ether.describe("plate + green-helix"),
								_ => ether.describe("_ + green-helix")
							};
						}
					}
				}
				// Conversation
				if (ether.mode == Mode::TalkSpeak) && ether.talker.eq(ent.id()) {
					if sys.poll_btn(Button::Left) {
						let t = "green-helix";
						match ether.sel_cue.as_str() {
							"greet polite" => ether.talk(t, "reply greet polite", &["greet impolite", "fear", "colors", "bye"]),
							"greet impolite" => ether.talk(t, "reply greet impolite", &["greet polite", "fear", "colors", "bye"]),
							"fear" => ether.talk(t, "reply fear", &[""; 0]),
							"colors" => ether.talk(t, "reply colors", &["colors try", "colors omit"]),
							"colors try" => ether.talk(t, "reply colors try", &["greet polite", "greet impolite", "fear", "bye"]),
							"colors omit" => ether.talk(t, "", &["greet polite", "greet impolite", "fear", "bye"]),
							_ => ether.talk(t, "", &[""; 0])
						}
					}
				}

				// Yellow helix
				let ent = &mut entities[i_yellow_hel];
				for ball in &mut ent.balls {
					ball.rad = (65536.0 * 900.0 * (1.0 + 0.25 * ((state.time as f64) * 1e-6 * 2.0 * PI).sin()) ) as i64;
				}
				match *yellow_hel_flying {
					false => {
						if ent.visible && ether.sync_reach(&state.plr_att.pos, &ent.att.pos, YELLOW_HELIX_REACH_DIST, ent.id()) {
							if (ether.mode == Mode::Fly) && ether.sel_reachable.eq(ent.id()) {
								if sys.poll_btn(Button::Right) {
									match state.yellow_helix_flew {
										false => ether.describe("yellow-helix before flight"),
										true => ether.describe("yellow-helix after flight")
									}
								} else if sys.poll_btn(Button::Left) {
									match state.sel_item.as_str() {
										"" => ether.describe("() + yellow-helix"),
										"melody" => ether.describe("melody + yellow-helix"),
										"plate" => {
											state.remove_item("plate")?;
											state.yellow_helix_flew = true;
											ether.describe("plate + yellow-helix");
											*yellow_hel_flying = true;
											*yellow_hel_flying_start_time = state.time;
										},
										_ => ether.describe("_ + yellow-helix")
									};
								}
							}
						}
					},
					true => {
						let traj = & cosm.trajectories[i_helix_flight];
						let time = state.time - (*yellow_hel_flying_start_time);
						if time <= traj.last_time() {
							ent.att = traj.attitude(time);
						} else {
							ent.att = traj.last_attitude();
							*yellow_hel_flying = false;
						}
					}
				}

				Ok(())
			},
			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

}
