/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use crate::master::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	Context,
	Master,
	Mode
};

const MID: &str = "Enter";

pub enum Phase {
	Pre,
	Segm1,
	Desc1to2,
	Segm2,
	Talk2to3,
	Segm3,
	Post
}

pub enum MEnter {
	Unset,
	Set {
		phase: Phase,
		phase_start_time: i128,
		i_segm1: usize,
		i_segm2: usize,
		i_segm3: usize
	}
}

impl MEnter {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}
}

impl Master for MEnter {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, Context{lingua, cosm, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				lingua.load(&["enter"])?;
				let phase_start_time: i128 = 0;
				let i_segm1 = cosm.trajectory_ind("enter-segment-1")?;
				let i_segm2 = cosm.trajectory_ind("enter-segment-2")?;
				let i_segm3 = cosm.trajectory_ind("enter-segment-3")?;
				let phase = match state.entered {
					false => Phase::Pre,
					true => {
						ether.mode = Mode::Fly;
						Phase::Post
					}
				};
				*self = Self::Set{phase, phase_start_time, i_segm1, i_segm2, i_segm3};
				Ok(())
			},
			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, Context{cosm, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Set{ref mut phase, ref mut phase_start_time, i_segm1, i_segm2, i_segm3} => {
				match *phase {
					Phase::Pre => {
						state.entered = true;
						*phase = Phase::Segm1;
						*phase_start_time = state.time;
					},
					Phase::Segm1 => {
						let traj = & cosm.trajectories[i_segm1];
						let time = state.time - (*phase_start_time);

						if time <= traj.last_time() {
							state.plr_att = traj.attitude(time);
						} else {
							state.plr_att = traj.last_attitude();
							ether.describe("enter 1to2");
							*phase = Phase::Desc1to2;
						}
					},
					Phase::Desc1to2 => {
						if ether.mode == Mode::Passive { // viewed all descriptions
							*phase = Phase::Segm2;
							*phase_start_time = state.time;
						}
					}
					Phase::Segm2 => {
						let traj = & cosm.trajectories[i_segm2];
						let time = state.time - (*phase_start_time);

						if time <= traj.last_time() {
							state.plr_att = traj.attitude(time);
						} else {
							state.plr_att = traj.last_attitude();
							ether.talk("unknown-voice", "greet", &[""; 0]);
							*phase = Phase::Talk2to3;
						}
					},
					Phase::Talk2to3 => {
						if ether.mode == Mode::Passive { // talk ended
							*phase = Phase::Segm3;
							*phase_start_time = state.time;
						}
					},
					Phase::Segm3 => {
						let traj = & cosm.trajectories[i_segm3];
						let time = state.time - (*phase_start_time);

						if time <= traj.last_time() {
							state.plr_att = traj.attitude(time);
						} else {
							state.plr_att = traj.last_attitude();
							ether.mode = Mode::Fly;
							*phase = Phase::Post;
						}
					},
					_ => {}
				}

				Ok(())
			},
			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

}
