/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use std::f64::consts::PI;

use crate::{
	base::RGB,
	sys::Key,
	cosm::CFont
};

use super::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	Context,
	Master
};

const MID: &str = "Telemetry";
const COLOR: RGB = RGB{r: 0xFF, g: 0xFF, b: 0xFF};
const SHADE: u8 = 0xE0;
const MARGIN: i32 = 0;

pub enum MTelemetry {
	Unset,
	Set {
		show: bool
	}
}

impl MTelemetry {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}
}

impl Master for MTelemetry {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, _ctx: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				let show = false;
				*self = Self::Set{show};
				Ok(())
			},
			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, Context{sys, ref fonts, cosm, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
		 	Self::Set{ref mut show, ..} => {
				if *show {
					let fh = fonts.get(CFont::System).height();
					let mut y: i32 = 0;
					let _ = sys.draw_text(fonts.get(CFont::System), format!("FPS = {:.3}", sys.fps()), 0, COLOR, SHADE, MARGIN, -(sys.width()>>1), -(sys.height()>>1)+y); y += fh;
					let _ = sys.draw_text(fonts.get(CFont::System), format!("CPUs = {}, FOV = {}°, MaxLookDist = {:.3}", sys.num_cpus(), sys.field_of_view(), ((1i64 << state.max_look_dist_binlog) as f64) / 65536.0), 0,  COLOR, SHADE, MARGIN, -(sys.width()>>1), -(sys.height()>>1)+y); y += fh;
					let _ = sys.draw_text(fonts.get(CFont::System), format!("chamber '{}', x = {:.3}, y = {:.3}, z = {:.3}", cosm.chambers[state.plr_i_chamber].id(), (state.plr_att.pos.x as f64) / 65536.0, (state.plr_att.pos.x as f64) / 65536.0, (state.plr_att.pos.z as f64) / 65536.0), 0, COLOR, SHADE, MARGIN, -(sys.width()>>1), -(sys.height()>>1)+y); y += fh;
					let _ = sys.draw_text(fonts.get(CFont::System), format!("yaw = {:.3}°, pitch = {:.3}°, roll = {:.3}°", state.plr_att.ori.yaw * 180.0 / PI, state.plr_att.ori.pitch * 180.0 / PI, state.plr_att.ori.roll * 180.0 / PI), 0, COLOR, SHADE, MARGIN, -(sys.width()>>1), -(sys.height()>>1)+y); y+= fh;
					let _ = sys.draw_text(fonts.get(CFont::System), format!("look-ray length = {:.3}", (ether.central_look_ray_dist as f64) / 65536.0), 0, COLOR, SHADE, MARGIN, -(sys.width()>>1), -(sys.height()>>1)+y);
				}
		        if sys.poll_key(Key::T) {
		            *show = !(*show);
		        }
				Ok(())
			},

			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

}
