/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use std::fs;

use crate::{
	base::{
		BASE_DIR,
		ARGB,
		RGB,
		Hash64
	},
	sys::{
		SOUNDS_DIR,
		SPEECHES_SUBDIR,
		Button,
		Image,
		Sound,
		System
	},
	cosm::CFont
};

use super::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	Context,
	Master,
	Mode
};

const MID: &str = "Talk";

const TALKER_ID_PREFIX: &str = "ent";
const TALKER_TEXT_COLOR: RGB = RGB{r: 0xD8, g: 0xD8, b: 0xF0};
const TALKER_TEXT_SHADE: u8 = 0xE8;
const TALKER_TEXT_MARGIN: i32 = 1;
const TALKER_FRAME_INT_COLOR: ARGB = ARGB{a: 0xF0, r: 0x20, g: 0x20, b: 0x30};
const TALKER_FRAME_EXT_COLOR: ARGB = ARGB{a: 0xE0, r: 0xB0, g: 0xB0, b: 0xE0};
const TALKER_FRAME_THICKNESS_BINLOG: i32 = 3;
const TALKER_FRAME_THICKNESS: i32 = 1 << TALKER_FRAME_THICKNESS_BINLOG;

const SPEECH_ID_PREFIX: &str = "speech";
const SPEECH_TEXT_COLOR: RGB = RGB{r: 0xE0, g: 0xE0, b: 0xF0};
const SPEECH_TEXT_SHADE: u8 = 0xD8;
const SPEECH_TEXT_MARGIN: i32 = 2;
const SPEECH_FRAME_INT_COLOR: ARGB = ARGB{a: 0xF0, r: 0x28, g: 0x28, b: 0x30};
const SPEECH_FRAME_EXT_COLOR: ARGB = ARGB{a: 0xE0, r: 0xC0, g: 0xC0, b: 0xE0};
const SPEECH_FRAME_THICKNESS_BINLOG: i32 = 3;
const SPEECH_FRAME_THICKNESS: i32 = 1 << SPEECH_FRAME_THICKNESS_BINLOG;

const CUE_ID_PREFIX: &str = "cue";
const CUE_TEXT_COLOR: RGB = RGB{r: 0xB8, g: 0xB8, b: 0xB0};
const CUE_TEXT_SEL_COLOR: RGB = RGB{r: 0xF0, g: 0xF0, b: 0xE0};
const CUE_TEXT_SHADE: u8 = 0xD8;
const CUE_TEXT_MARGIN: i32 = 2;
const CUE_FRAME_INT_COLOR: ARGB = ARGB{a: 0xF0, r: 0x30, g: 0x30, b: 0x28};
const CUE_FRAME_EXT_COLOR: ARGB = ARGB{a: 0xE0, r: 0xE0, g: 0xE0, b: 0xC0};
const CUE_FRAME_THICKNESS_BINLOG: i32 = 3;
const CUE_FRAME_THICKNESS: i32 = 1 << CUE_FRAME_THICKNESS_BINLOG;

pub enum MTalk {
	Unset,
	Set {
		curr_speech_i: i32,
		curr_speech_txt: String,
		curr_speech_res_img: Result<Image, String>, // "cache"
		speech_sounds: Vec<Option<Sound>>,
		cues_txt: Vec<String>,
		cues_res_imgs: Vec<[Result<Image, String>; 2]> // another cache, 2 images (selected and non-selected) for each cue. Order is reversed
	}
}

fn load_speech_sounds(Context{sys, ..}: &mut Context) -> Vec<Option<Sound>> {
	let mut sounds = Vec::<Option<Sound>>::new();

	if let Ok(entry_iter) = fs::read_dir(format!("{}/{}/{}", BASE_DIR, SOUNDS_DIR, SPEECHES_SUBDIR)) {
		for entry in entry_iter {
			if let Ok(entry) = entry {
				if let Ok(e_type) = entry.file_type() {
					if e_type.is_file() {
						let filename = entry.file_name().into_string();
						if let Ok(filename) = filename {
							sounds.push(sys.load_speech(filename)); // all files in that dir should be sounds, loadable by sdl-mixer
						}
					}
				}
			}
		}
	}

	sounds
}

impl MTalk {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}

}

impl Master for MTalk {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, ctx: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				let curr_speech_i: i32 = 0;
				let curr_speech_txt = String::new();
				let curr_speech_res_img = System::make_text_image(ctx.fonts.get(CFont::Talk_Speech), &curr_speech_txt, 0, RGB{r: 0, g: 0, b: 0});
				let speech_sounds = load_speech_sounds(ctx);
				let cues_txt = Vec::<String>::new();
				let cues_res_imgs = Vec::<[Result<Image, String>; 2]>::new();
				*self = Self::Set{curr_speech_i, curr_speech_txt, curr_speech_res_img, speech_sounds, cues_txt, cues_res_imgs};
				Ok(())
			},
			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, Context{sys, ref fonts, lingua, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
		 	Self::Set{ref mut curr_speech_i, ref mut curr_speech_txt, ref mut curr_speech_res_img, ref speech_sounds, ref mut cues_txt, ref mut cues_res_imgs} => {
				// Draw (and start playing sound...) if drawing unlocked

				if !sys.draw_locked() {
					let mut y: i32 = -(sys.height() >> 1);

					// Talker's name
					match ether.mode { Mode::TalkListen | Mode::TalkSpeak => {
						let txt = lingua.get_or_echo(format!("{} {}", TALKER_ID_PREFIX, & ether.talker));
						let (xmt, ymt) = (- (sys.width() >> 1) + TALKER_FRAME_THICKNESS, - (sys.height() >> 1) + TALKER_FRAME_THICKNESS);
						let mut hmt = TALKER_TEXT_MARGIN << 1;
						if let Ok(txt_img) = System::make_text_image(fonts.get(CFont::Talk_Talker), &txt, 0, TALKER_TEXT_COLOR) {
							hmt += txt_img.height();
							let _ = sys.draw_rect(xmt, ymt, (sys.width() >> 1) - TALKER_FRAME_THICKNESS - 1, ymt + hmt - 1, ARGB{a: TALKER_TEXT_SHADE, r: 0, g: 0, b: 0}, true);
							let _ = sys.draw_text_image(&txt_img, 0, TALKER_TEXT_MARGIN, - (txt_img.width() >> 1) - TALKER_TEXT_MARGIN, ymt);
						}
						let _ = sys.draw_frame(xmt, ymt, sys.width() - (TALKER_FRAME_THICKNESS << 1), hmt, TALKER_FRAME_INT_COLOR, TALKER_FRAME_EXT_COLOR, TALKER_FRAME_THICKNESS_BINLOG);
						y += hmt + (TALKER_FRAME_THICKNESS << 1);
					}, _ => () }

					// Talker's speech
					if ether.mode == Mode::TalkListen {
						// Check if it has just started
						*curr_speech_i = (*curr_speech_i).max(1);
						match lingua.get(format!("{} {} {} {}", SPEECH_ID_PREFIX, & ether.talker, & ether.speech, *curr_speech_i)) {
							Some(txt) => {
								// "Cache"
								if !txt.eq(curr_speech_txt) { // new speech to make its (wrapped) image... and play "its" sound
									*curr_speech_txt = txt;
									*curr_speech_res_img = System::make_text_image(fonts.get(CFont::Talk_Speech), &curr_speech_txt, sys.width() - ((SPEECH_FRAME_THICKNESS + SPEECH_TEXT_MARGIN) << 1), SPEECH_TEXT_COLOR);

									if speech_sounds.len() > 0 {
										let _ = sys.play_speech(speech_sounds[ (curr_speech_txt.hash64() as usize) % speech_sounds.len() ].as_ref()); // "same speech - same hash - same sound"
									}
								}
								let (xmt, ymt) = (- (sys.width() >> 1) + SPEECH_FRAME_THICKNESS, y + SPEECH_FRAME_THICKNESS);
								let mut hmt = SPEECH_TEXT_MARGIN << 1;
								if let Ok(ref txt_img) = curr_speech_res_img {
									hmt += txt_img.height();
									let _ = sys.draw_rect(xmt, ymt, (sys.width() >> 1) - SPEECH_FRAME_THICKNESS - 1, ymt + hmt - 1, ARGB{a: SPEECH_TEXT_SHADE, r: 0, g: 0, b: 0}, true);
									let _ = sys.draw_text_image(&txt_img, 0, SPEECH_TEXT_MARGIN, xmt, ymt);
								}
								let _ = sys.draw_frame(xmt, ymt, sys.width() - (SPEECH_FRAME_THICKNESS << 1), hmt, SPEECH_FRAME_INT_COLOR, SPEECH_FRAME_EXT_COLOR, SPEECH_FRAME_THICKNESS_BINLOG);
							},

							None => { // no more speeches
								*curr_speech_i = 0; // for next sequence of speeches
								*curr_speech_txt = String::new();  // for next sequence of speeches (esp. sound, see "cache" above)
								ether.mode = match ether.cues.len() {
									0 => ether.pre_talk_mode, // Fly, Passive, or?..
									_ => Mode::TalkSpeak
								};

								let _ = sys.play_speech(None); // stop/fade out currently played speech, if any
							}
						}
					}

					// Player's cues
					if ether.mode == Mode::TalkSpeak {
						let xmt = - (sys.width() >> 1) + CUE_FRAME_THICKNESS;
						let mut ymt = (sys.height() >> 1) - CUE_FRAME_THICKNESS;
						let mut hmt: i32 = 0;

						for (i, c) in ether.cues.iter().rev().enumerate() {
							ymt -= CUE_TEXT_MARGIN << 1;
							hmt += CUE_TEXT_MARGIN << 1;
							let txt = lingua.get_or_echo(format!("{} {} {}", CUE_ID_PREFIX, & ether.talker, c));

							// Cache
							if cues_txt.len() == i {
								cues_txt.push(String::new());
								cues_res_imgs.push([
									System::make_text_image(fonts.get(CFont::Talk_Cue), "", 0, RGB{r: 0, g: 0, b: 0}),
									System::make_text_image(fonts.get(CFont::Talk_Cue), "", 0, RGB{r: 0, g: 0, b: 0})
								]);
							}
							if !txt.eq(&(cues_txt[i])) {
								cues_txt[i] = txt;
								cues_res_imgs[i] = [
									System::make_text_image(fonts.get(CFont::Talk_Cue), & cues_txt[i], sys.width() - ((CUE_FRAME_THICKNESS + CUE_TEXT_MARGIN) << 1), CUE_TEXT_COLOR),
									System::make_text_image(fonts.get(CFont::Talk_Cue), & cues_txt[i], sys.width() - ((CUE_FRAME_THICKNESS + CUE_TEXT_MARGIN) << 1), CUE_TEXT_SEL_COLOR)
								];
							}
							let j: usize = match ether.sel_cue.eq(c) {
								true => 1,
								_ => 0
							};
							if let Ok(ref txt_img) = & cues_res_imgs[i][j] {
								ymt -= txt_img.height();
								hmt += txt_img.height();
								let _ = sys.draw_rect(xmt, ymt, (sys.width() >> 1) - CUE_FRAME_THICKNESS - 1, ymt + txt_img.height() + (CUE_TEXT_MARGIN << 1) - 1, ARGB{a: CUE_TEXT_SHADE, r: 0, g: 0, b: 0}, true);
								let _ = sys.draw_text_image(&txt_img, 0, CUE_TEXT_MARGIN, xmt, ymt);
							}
						}

						let _ = sys.draw_frame(xmt, ymt, sys.width() - (CUE_FRAME_THICKNESS << 1), hmt, CUE_FRAME_INT_COLOR, CUE_FRAME_EXT_COLOR, CUE_FRAME_THICKNESS_BINLOG);
					}

				}

				// Process input regardless of drawing lock

				match ether.mode {
					Mode::TalkSpeak => {
						if !sys.test_modkeys() {
							let mut di: i32 = 0;
							if sys.mouse_dw() != 0 {
								di = - sys.mouse_dw();
							}/* else if sys.poll_keys(&[Key::Up, Key::Kp8]) {
								di = -1;
							} else if sys.poll_keys(&[Key::Down, Key::Kp2]) {
								di = 1;
							} else if sys.poll_keys(&[Key::PageUp, Key::Kp9, Key::Home, Key::Kp7]) {
								di = - (ether.cues.len() as i32);
							} else if sys.poll_keys(&[Key::PageDown, Key::Kp3, Key::End, Key::Kp1]) {
								di = ether.cues.len() as i32;
							}*/
							// Is it more or less convenient to be able to choose cue also by keyboard...
							// ...what if someone falls into "choose by wheel? choose by keys? by wheel? by keys?"
							// So, comment this block? uncomment it? comment? uncomment?

							if di != 0 {
								let mut i = match ether.sel_cue_ind() {
									Some(i) => i as i32,
									None => 0
								};
								i = (i + di).max(0).min((ether.cues.len() - 1) as i32);
								ether.sel_cue = ether.cues[i as usize].clone();
							}
						}
					},

					Mode::TalkListen => {
						// Switch to next speech
						if sys.poll_btn(Button::Right) {
							*curr_speech_i += 1;
						}
					},

					_ => ()
				}

				Ok(())
			},

			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

}
