/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use crate::{
	base::PrependErrorString,
	state::AUTOSAVE_FILEPATH,
};

use super::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	ERR_NOT_SET_CANNOT_FINISH,
	Context,
	Master
};

const MID: &str = "Sync";

const AUTOSAVE_INTERVAL: i128 = 60 * 1_000_000; // each play-time minute

pub enum MSync {
	Unset,
	Set {
		time_before_last_pause: i128,
		last_resume_moment: i128,
		last_autosave_time: i128,
	}
}

impl MSync {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}
}

impl Master for MSync {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, Context{sys, state, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				*self = Self::Set {
					time_before_last_pause: state.time,
					last_resume_moment: sys.now(),
					last_autosave_time: state.time
				};

				sys.resume_audio();

				Ok(())
			},
			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, Context{sys, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Set{ref mut time_before_last_pause, ref mut last_resume_moment, ref mut last_autosave_time} => {
				if ether.paused { // resume
					*time_before_last_pause = state.time;
					*last_resume_moment = sys.now();
					ether.paused = false;

					sys.resume_audio();
				}

				sys.sync()?;

				let time_after_last_resume = sys.now() - (*last_resume_moment);

				state.time = (*time_before_last_pause) + time_after_last_resume;

				if state.time - (*last_autosave_time) > AUTOSAVE_INTERVAL {
					state.save(AUTOSAVE_FILEPATH).pre_err("cannot save state to autosave")?;
					*last_autosave_time = state.time;
				}

				Ok(())
			},
			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

	fn finish(&mut self, Context{sys, state, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Set{..} => {
				if state.transversed {
					state.prev_verse = state.verse;
					state.verse = state.next_verse;
				}

				sys.halt_audio();

				Ok(())
			},
			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_FINISH))
		}
	}

}
