/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate rayon;

use std::f64::consts::PI;

use rayon::prelude::*;

use crate::{
	base,
	base::V3Di,
	cosm::Texture
};

use super::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	Context,
	Master
};

const MID: &str = "Render-Walls";

const MAX_RAY_STEPS: i32 = 1 << 10; // in case ray tracing stucks...

pub enum MRenderWalls {
	Unset,
	Set {
		scr_width: i32,
		scr_height: i32,
		look_rays: Vec<Vec<V3Di>>, // rays of unit length from origin to screen points
		scr_rays_z_cos: Vec<Vec<i64>>, // cosine of angle between eye-to-screen ray and its projection (of length 1) onto player's Ox
		reciprocals: Vec<i64>, // "1/x for x in [-2, 2)" in FPA(0x10)
		pixel_fn: fn(&Texture, i64, i64, i64, u8) -> [u8; 4] // pixel (nearest, faster) or pixel_smooth (interpolate, slower)
	}
}

impl MRenderWalls {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}
}

/*
#[inline(always)]
fn four_are_nonnegative(a: i64, b: i64, c: i64, d: i64) -> bool {
	(((a as u64) | (b as u64) | (c as u64) | (d as u64)) & 0x8000_0000_0000_0000) == 0 // all sign bits are 0
}
*/

impl Master for MRenderWalls {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, Context{sys, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				// "Screen" is at unit distance from origin and orthogonal to Ox
				let fov: f64 = (sys.field_of_view() as f64) * PI / 180.0;
				let width: f64 = 2.0 * ((0.5 * fov).tan());
				let height: f64 = width * (sys.height() as f64) / (sys.width() as f64);

				let scr_width = if sys.antialiasing_4x() { sys.width() << 1 } else { sys.width() };
				let scr_height = if sys.antialiasing_4x() { sys.height() << 1 } else { sys.height() };

				let mut look_rays = Vec::<Vec<V3Di>>::with_capacity(scr_height as usize);
				let mut scr_rays_z_cos = Vec::<Vec<i64>>::with_capacity(scr_height as usize);

				let sx: f64 = 1.0;
				for y in 0..scr_height {
					let sz: f64 = (0.5 - ((y as f64) - 0.5) / (scr_height as f64)) * height; // -0.5 for symmetry of upper/lower halves' rays
					let mut look_rays_row = Vec::<V3Di>::with_capacity(scr_width as usize);
					let mut scr_rays_z_cos_row = Vec::<i64>::with_capacity(scr_width as usize);
					for x in 0..scr_width {
						let sy: f64 = (0.5 - ((x as f64) - 0.5) / (scr_width as f64)) * width; // -0.5 for symmetry of right/left halves' rays

						let l: f64 = (sx * sx + sy * sy + sz * sz).sqrt();

						look_rays_row.push(V3Di{
							x: ((sx * 65536.0 / l) as i64).max(-0x10000).min(0x10000),
							y: ((sy * 65536.0 / l) as i64).max(-0x10000).min(0x10000),
							z: ((sz * 65536.0 / l) as i64).max(-0x10000).min(0x10000)
						});
						scr_rays_z_cos_row.push((65536.0 / l) as i64);
					}
					look_rays.push(look_rays_row);
					scr_rays_z_cos.push(scr_rays_z_cos_row);
				}

				let mut reciprocals = Vec::<i64>::with_capacity(0x40000);
				for x in -0x20000i64..0x20000i64 { // [-2, 2) to deal with slightly out of [-1, 1] values
					reciprocals.push(if x != 0 { ((0x1_0000_0000i64 as f64) / (x as f64)) as i64 } else { 0 }); // "1/x" in fixed point arithmetic
				}

				let pixel_fn = if sys.texture_smooth() { Texture::pixel_smooth } else { Texture::pixel };

				*self = Self::Set{scr_width, scr_height, look_rays, scr_rays_z_cos, reciprocals, pixel_fn};
				Ok(())
			},

			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, Context{sys, cosm, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Set{scr_width, scr_height, ref look_rays, ref scr_rays_z_cos, ref reciprocals, pixel_fn} => {
				if !sys.draw_locked() { // only when new frame is required, to decrease load
					let chambers = &(cosm.chambers);
					let textures = &(cosm.textures);
					let supra_textures = &(cosm.supra_textures);

					let ch0 =  &(chambers[state.plr_i_chamber]);

					let p0 = state.plr_att.pos.subbed(&V3Di{x: ch0.x, y: ch0.y, z: ch0.z}); // player's position in current chamber's RF

					let max_dist_binlog = state.max_look_dist_binlog();
					let max_dist = 1i64 << max_dist_binlog;

					let (yaw_cos, yaw_sin,
						pitch_cos, pitch_sin,
						roll_cos, roll_sin) = state.plr_att.ori.cosines_sines_fpa();

				    // Parallelization
					let vb_rows = sys.screen_raw_chunks_mut(scr_width);
					let zb_rows = ether.zbuf_raw_chunks_mut(scr_width);
					let rows: Vec<(usize, (&mut [u8], &mut [i64]))> = vb_rows.zip(zb_rows).enumerate().collect();
					rows.into_par_iter().for_each(|(y, (vb_row, zb_row))| { // rayon parallel iterator
// Executed by each thread
// Presumably the most computationally intensive part... optimizations needed...
// Fixed point arithmetic (FPA), shift = 0x10
						let mut offs: usize = 0;

					    for x in 0..scr_width { // x is "screen x in pixels"
						   	 let mut color: [u8; 4] = [0; 4]; // black

						   	 let ray = look_rays[y][x as usize]
						   		 .rolled(roll_cos, roll_sin)
						   		 .pitched(pitch_cos, pitch_sin)
						   		 .yawed(yaw_cos, yaw_sin);

						   	 let ray_inv = V3Di{// & 0x3FFFF to avoid index out of bounds due to inaccuracies
						   		 x: reciprocals[((0x20000 + ray.x) & 0x3FFFF) as usize],
						   		 y: reciprocals[((0x20000 + ray.y) & 0x3FFFF) as usize],
						   		 z: reciprocals[((0x20000 + ray.z) & 0x3FFFF) as usize]
						   	 };

						   	 let mut ch = ch0;
						   	 let mut p = p0.clone();

						   	 let mut dist: i64 = 0;
						   	 let mut ray_steps_left: i32 = MAX_RAY_STEPS;

						   	 // Trace the ray until it hits chamber face and there is no adjacent chamber at this point
						   	 // or maximum number of steps is reached
						   	 // or distance is greater than maximum trace distance
						   	 while ray_steps_left > 0 {
						   		 ray_steps_left -= 1;
						   		 // Determine times to reach 3 planes through chamber faces: parallel to Oxy, Oyz, Ozx
						   		 // Each plane is chosen from two depending on sign of ray's x/y/z component

						   		 let t = V3Di{
						   			 x: match ray.x.signum() {
						   				 -1 => (((ray_inv.x as i128) * ((- p.x) as i128)) >> 0x10) as i64,
						   				 1 => (((ray_inv.x as i128) * ((ch.lx - p.x) as i128)) >> 0x10) as i64,
						   				 _ => std::i64::MAX // "1/0 = ∞"
						   			 },
						   			 y: match ray.y.signum() {
						   				 -1 => (((ray_inv.y as i128) * ((- p.y) as i128)) >> 0x10) as i64,
						   				 1 => (((ray_inv.y as i128) * ((ch.ly - p.y) as i128)) >> 0x10) as i64,
						   				 _ => std::i64::MAX // "1/0 = ∞"
						   			 },
						   			 z: match ray.z.signum() {
						   				 -1 => (((ray_inv.z as i128) * ((- p.z) as i128)) >> 0x10) as i64,
						   				 1 => (((ray_inv.z as i128) * ((ch.lz - p.z) as i128)) >> 0x10) as i64,
						   				 _ => std::i64::MAX // "1/0 = ∞"
						   			 }
						   		 };

						   		 let mut h = V3Di{x: 0, y: 0, z: 0}; // hit point (may be inaccurate...) in current chamber's reference frame (RF)

						   		 let mut further: bool = false;

						   		 // Minimal time and sign of ray component determine chamber face the ray actually hits
						   		 // Check if there is an adjacent chamber at hit point,
						   		 // recalculating hit point coordinates in adjacent chamber's RF
						   		 // which are the coordinates of the point on traced ray at next iteration

						   		 if t.z <= t.x {
						   			 if t.z <= t.y { // t.z is min
						   				 dist += t.z;
						   				 if dist > max_dist {
						   					 dist = max_dist;
						   					 break; // color is black
						   				 }
						   				 h.x = p.x + ((((t.z as i128) * (ray.x as i128)) >> 0x10) as i64);
						   				 h.y = p.y + ((((t.z as i128) * (ray.y as i128)) >> 0x10) as i64);
						   				 let face = if ray.z > 0 { 3 } else { 2 };
						   				 for i in &ch.adj[face] {
						   					 let adj_ch = &(chambers[*i]);
						   					 p.x = h.x + (ch.x - adj_ch.x);
						   					 p.y = h.y + (ch.y - adj_ch.y);
						   					 if (p.x >= 0) && (p.y >= 0) && (p.x <= adj_ch.lx) && (p.y <= adj_ch.ly) {
						   						 p.z = if face == 2 { adj_ch.lz } else { 0 };
						   						 ch = adj_ch;
						   						 further = true;
						   						 break;
						   					 }
						   				 }
						   				 if !further {
						   					 let (cell, rotm) = supra_textures[ch.sup_txr[face]].cell(h.x, h.y);
						   					 color = pixel_fn(&(textures[cell]), h.x, h.y, dist, rotm); // textures[cell].pixel(h.x, h.y, dist, refl);
						   					 let dist_to_max = max_dist - dist;
						   					 for c in 0..3 {
						   						 color[c] = base::scale_u8(color[c], dist_to_max, max_dist_binlog);
						   					 }
						   					 break;
						   				 }
						   			 } else { // t.y is min
						   				 dist += t.y;
						   				 if dist > max_dist {
						   					 dist = max_dist;
						   					 break; // color is black
						   				 }
						   				 h.z = p.z + ((((t.y as i128) * (ray.z as i128)) >> 0x10) as i64);
						   				 h.x = p.x + ((((t.y as i128) * (ray.x as i128)) >> 0x10) as i64);
						   				 let face = if ray.y > 0 { 4 } else { 1 };
						   				 for i in &ch.adj[face] {
						   					 let adj_ch = &(chambers[*i]);
						   					 p.x = h.x + (ch.x - adj_ch.x);
						   					 p.z = h.z + (ch.z - adj_ch.z);
						   					 if (p.x >= 0) && (p.z >= 0) && (p.x <= adj_ch.lx) && (p.z <= adj_ch.lz) {
						   						 p.y = if face == 1 { adj_ch.ly } else { 0 };
						   						 ch = adj_ch;
						   						 further = true;
						   						 break;
						   					 }
						   				 }
						   				 if !further {
						   					 let (cell, rotm) = supra_textures[ch.sup_txr[face]].cell(h.z, h.x);
						   					 color = pixel_fn(&(textures[cell]), h.z, h.x, dist, rotm); // textures[cell].pixel(h.z, h.x, dist, refl);
						   					 let dist_to_max = max_dist - dist;
						   					 for c in 0..3 {
						   						 color[c] = base::scale_u8(color[c], dist_to_max, max_dist_binlog);
						   					 }
						   					 break;
						   				 }
						   			 }
						   		 } else {
						   			 if t.x <= t.y { // t.x is min
						   				 dist += t.x;
						   				 if dist > max_dist {
						   					 dist = max_dist;
						   					 break; // color is black
						   				 }
						   				 h.y = p.y + ((((t.x as i128) * (ray.y as i128)) >> 0x10) as i64);
						   				 h.z = p.z + ((((t.x as i128) * (ray.z as i128)) >> 0x10) as i64);
						   				 let face = if ray.x > 0 { 5 } else { 0 };
						   				 for i in &ch.adj[face] {
						   					 let adj_ch = &(chambers[*i]);
						   					 p.y = h.y + (ch.y - adj_ch.y);
						   					 p.z = h.z + (ch.z - adj_ch.z);
						   					 if (p.y >= 0) && (p.z >= 0) && (p.y <= adj_ch.ly) && (p.z <= adj_ch.lz) {
						   						 p.x = if face == 0 { adj_ch.lx } else { 0 };
						   						 ch = adj_ch;
						   						 further = true;
						   						 break;
						   					 }
						   				 }
						   				 if !further {
						   					 let (cell, rotm) = supra_textures[ch.sup_txr[face]].cell(h.y, h.z);
						   					 color = pixel_fn(&(textures[cell]), h.y, h.z, dist, rotm); // textures[cell].pixel(h.y, h.z, dist, refl);
						   					 let dist_to_max = max_dist - dist;
						   					 for c in 0..3 {
						   						 color[c] = base::scale_u8(color[c], dist_to_max, max_dist_binlog);
						   					 }
						   					 break;
						   				 }
						   			 } else { // t.y is min... duplicated
						   				 dist += t.y;
						   				 if dist > max_dist {
						   					 dist = max_dist;
						   					 break; // color is black
						   				 }
						   				 h.z = p.z + ((((t.y as i128) * (ray.z as i128)) >> 0x10) as i64);
						   				 h.x = p.x + ((((t.y as i128) * (ray.x as i128)) >> 0x10) as i64);
						   				 let face = if ray.y > 0 { 4 } else { 1 };
						   				 for i in &ch.adj[face] {
						   					 let adj_ch = &(chambers[*i]);
						   					 p.x = h.x + (ch.x - adj_ch.x);
						   					 p.z = h.z + (ch.z - adj_ch.z);
						   					 if (p.x >= 0) && (p.z >= 0) && (p.x <= adj_ch.lx) && (p.z <= adj_ch.lz) {
						   						 p.y = if face == 1 { adj_ch.ly } else { 0 };
						   						 ch = adj_ch;
						   						 further = true;
						   						 break;
						   					 }
						   				 }
						   				 if !further {
						   					 let (cell, rotm) = supra_textures[ch.sup_txr[face]].cell(h.z, h.x);
						   					 color = pixel_fn(&(textures[cell]), h.z, h.x, dist, rotm); // textures[cell].pixel(h.z, h.x, dist, refl);
						   					 let dist_to_max = max_dist - dist;
						   					 for c in 0..3 {
						   						 color[c] = base::scale_u8(color[c], dist_to_max, max_dist_binlog);
						   					 }
						   					 break;
						   				 }
						   			 }
						   		 }

						   	 }

						   	 let offs4 = offs << 2;
						   	 for c in 0..3 {
						   		 vb_row[offs4 + c] = color[c];
						   	 }

							 // Recalc dist for render-balls, by projecting onto player's Ox
						   	 zb_row[offs] = (((dist as i128) * (scr_rays_z_cos[y][x as usize] as i128)) >> 0x10) as i64;

						   	 offs += 1;
					    }

					});

					ether.central_look_ray_dist = ether.z_buffer[((scr_height >> 1) * scr_width + (scr_width >> 1)) as usize];
				}

				Ok(())
			},

			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

}
