/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate rayon;

use std::f64::consts::PI;

use rayon::prelude::*;

use crate::{
	base,
	cosm::MAX_ENTITY_SIZE
};

use super::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	Context,
	Master
};

const MID: &str = "Render-Balls";

const MAX_BALL_RENDER_DIST: i64 = 1i64 << (0x10 + 20);
const MAX_BALL_SCREEN_DIAMETER: usize = 0x200;

pub enum MRenderBalls {
	Unset,
	Set {
		scr_width: i32, // in pixels
		scr_height: i32, // in pixels
		scr_dist: i64, // distance from "eye" to screen in pixels, in fixed point arithmetic (shift = 0x10)
		proto_disk: Vec<Vec<(i64, u8)>>, // z-delta & color-mult
		scale_to_proto: Vec<Vec<usize>> // precalculated scaled coordinates
	}
}

impl MRenderBalls {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}
}

impl Master for MRenderBalls {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, Context{sys, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				let scr_width = if sys.antialiasing_4x() { sys.width() << 1 } else { sys.width() };
				let scr_height = if sys.antialiasing_4x() { sys.height() << 1 } else { sys.height() };
				let scr_dist = ((0.5 * (scr_width as f64) / (0.5 * (sys.field_of_view() as f64) * PI / 180.0).tan()) as i64) << 0x10;

				let max_ball_screen_radius = 0.5 * (MAX_BALL_SCREEN_DIAMETER as f64);

				let mut proto_disk = Vec::<Vec<(i64, u8)>>::with_capacity(MAX_BALL_SCREEN_DIAMETER);
				for y in 0..MAX_BALL_SCREEN_DIAMETER {
					let mut proto_disk_row = Vec::<(i64, u8)>::with_capacity(MAX_BALL_SCREEN_DIAMETER);
					let v = ((y as f64) - max_ball_screen_radius) / max_ball_screen_radius;
					for x in 0..MAX_BALL_SCREEN_DIAMETER {
						let u = ((x as f64) - max_ball_screen_radius) / max_ball_screen_radius;
						let w = 1.0 - u * u - v * v;
						if w >= 0.0 {
							let sw = w.sqrt();
							proto_disk_row.push(((-sw * 65536.0) as i64, ((0.5 + 0.5 * sw) * 255.0) as u8));
						} else {
							proto_disk_row.push((std::i64::MAX, 0));
						}
					}
					proto_disk.push(proto_disk_row);
				}

				let mut scale_to_proto = Vec::<Vec<usize>>::with_capacity(MAX_BALL_SCREEN_DIAMETER + 1);
				for i in 0..=MAX_BALL_SCREEN_DIAMETER {
					let mut coords = Vec::<usize>::with_capacity(i);
					for j in 0..i {
						coords.push(j * MAX_BALL_SCREEN_DIAMETER / i);
					}
					scale_to_proto.push(coords);
				}

				*self = Self::Set{scr_width, scr_height, scr_dist, proto_disk, scale_to_proto};
				Ok(())
			},

			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

	/// Another computationally intensive part, optimizations needed...
	/// Fixed point arithmetic (FPA), shift = 0x10
    fn run(&mut self, Context{sys, cosm, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Set{scr_width, scr_height, scr_dist, ref proto_disk, ref scale_to_proto} => {
				if !sys.draw_locked() { // only when new frame is required, to decrease load

					let scale_color_binlog = state.max_look_dist_binlog() + 8;
					let max_look_dist = 1i64 << state.max_look_dist_binlog();

					let max_ent_dist = (max_look_dist + MAX_ENTITY_SIZE).min(MAX_BALL_RENDER_DIST);
					let max_ent_dist_square = (max_ent_dist as i128) * (max_ent_dist as i128);

					let (plr_yaw_cos, plr_yaw_sin,
						plr_pitch_cos, plr_pitch_sin,
						plr_roll_cos, plr_roll_sin) = state.plr_att.ori.cosines_sines_fpa();

					let vb_rows = sys.screen_raw_chunks_mut(scr_width);
					let zb_rows = ether.zbuf_raw_chunks_mut(scr_width);
					let mut scr_rows: Vec<(usize, (&mut [u8], &mut [i64]))> = vb_rows.zip(zb_rows)
						.enumerate()
						.collect();

					let scr_half_width = (scr_width >> 1) as i64;
					let scr_half_height = (scr_height >> 1) as i64;

					for ent in &cosm.entities {
						if ent.visible {
							let d = ent.att.pos.subbed(&state.plr_att.pos); // entity pivot in RF with O=player and "absolute" orientation
							let pd = d
								.yawed_noverflow(plr_yaw_cos, -plr_yaw_sin)
								.pitched_noverflow(plr_pitch_cos, -plr_pitch_sin)
								.rolled_noverflow(plr_roll_cos, -plr_roll_sin); // entity pivot in player's RF

							if (d.norm_square() <= max_ent_dist_square) && (pd.x > -MAX_ENTITY_SIZE) { // not too far and not behind player's "eye"
								let (ent_yaw_cos, ent_yaw_sin,
									ent_pitch_cos, ent_pitch_sin,
									ent_roll_cos, ent_roll_sin
								) = ent.att.ori.cosines_sines_fpa();

								for ball in &ent.balls {
									let c = ball.pos
										.rolled_noverflow(ent_roll_cos, ent_roll_sin)
										.pitched_noverflow(ent_pitch_cos, ent_pitch_sin)
										.yawed_noverflow(ent_yaw_cos, ent_yaw_sin) // ball's center in RF with O=entity and "absolute" orientation...
										.added(&d) // ...in RF with O=player and "absolute" orientation...
										.yawed_noverflow(plr_yaw_cos, -plr_yaw_sin)
										.pitched_noverflow(plr_pitch_cos, -plr_pitch_sin)
										.rolled_noverflow(plr_roll_cos, -plr_roll_sin); // ...in player's RF

									if c.x > 0 { // not behind, can be visible
										let dist_scale_mult = ((scr_dist << 0x10) / c.x) as i128;
										let sdiam = (((((ball.rad << 1) as i128) * dist_scale_mult) >> 0x20) as i64).min(MAX_BALL_SCREEN_DIAMETER as i64); // screen diameter, in pixels
										if sdiam > 0 { // ball is big enough to be seen
											let sr = sdiam >> 1;
											// screen coordinates, in pixels, of top left corner of square containing projection-disk
											let sx = scr_half_width + ((-((c.y as i128) * dist_scale_mult) >> 0x20) as i64) - sr;
											let sy = scr_half_height + ((-((c.z as i128) * dist_scale_mult) >> 0x20) as i64) - sr;

											// Can projection-disk intersect screen?
											if (sx < (scr_width as i64)) && (sy < (scr_height as i64)) && (sx + sdiam >= 0) && (sy + sdiam >= 0) {
												// After all these checks... it must be actually drawn
												let (sx, sy, sdiam) = (sx as i32, sy as i32, sdiam as i32);

												let scale_to_proto = &(scale_to_proto[sdiam as usize]);
												let rad = ball.rad;
												let color = &(ball.color);

												// Corners of intersection with screen in square-with-disk's RF
												let (u0, v0) = (0i32.max(-sx), 0i32.max(-sy));
												let (u1, v1) = (sdiam.min(scr_width - sx), sdiam.min(scr_height - sy));

												let (y0, y1) = ((sy + v0) as usize, (sy + v1) as usize); // non-negative

												let i2v_diff: i32 = v0 - (y0 as i32);

												let scr_sub_rows = &mut scr_rows[y0..y1];
												// Parallelization by rayon iterator
												scr_sub_rows.into_par_iter().for_each(|(i, (vb_row, zb_row))| {
													let v = (*i as i32) + i2v_diff;
													let proto_row = &(proto_disk[scale_to_proto[v as usize]]);
													for u in u0..u1 {
														let proto = &(proto_row[scale_to_proto[u as usize]]);
														let dz = proto.0;
														if dz <= 0 { // pixel inside disk
															let offs = (sx + u) as usize;
															let z = c.x + ((dz * rad) >> 0x10);
															if z < zb_row[offs] { // is not occluded
																zb_row[offs] = z;
																let offs4 = offs << 2;
																let dist_to_max = max_look_dist - z;
																for k in 0..3 {
																	vb_row[offs4 + k] = base::scale_u8(color[k], (proto.1 as i64) * dist_to_max, scale_color_binlog);
																}
															}
														}
													}
												});

											}
										}
									}
								}
							}
						}

					}
				}
				// all these } mean... mean...

				Ok(())
			},

			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

}
