/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use std::f64::consts::PI;

use crate::{
	base::{
		Orientation,
		V3Df,
		V3Di
	},
	sys::{
		Button,
		Key
	}
};

use super::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	Context,
	Master,
	Mode
};

const MID: &str = "Movement";

const SHORT_STEP_DISTANCE: i64 = 0x200i64 << 0x10; // "1 meter"?
const LONG_STEP_DISTANCE: i64 = SHORT_STEP_DISTANCE * 10; // "10 meters"?
const ROTATION_MOUSE_SENSITIVITY_MULT: f64 = 0.5;

pub enum MMovement {
	Unset,
	Set
}

impl MMovement {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}
}

impl Master for MMovement {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, Context{cosm, state, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				// Sanity checks...
				if state.plr_i_chamber < cosm.chambers.len() {
					let chamber = &(cosm.chambers[state.plr_i_chamber]);
					if chamber.inside(&state.plr_att.pos) {
						*self = Self::Set;
						Ok(())
					} else {
						Err(format!("inconsistent state: player's position ({}, {}, {}) is not in player's chamber '{}' ({}+{}, {}+{}, {}+{})",
							state.plr_att.pos.x, state.plr_att.pos.y, state.plr_att.pos.z, chamber.id(),
							chamber.x, chamber.lx, chamber.y, chamber.ly, chamber.z, chamber.lz))
					}
				} else {
					Err(format!("inconsistent state: index of player's chamber {} is out of bounds ({} chambers in total)",
						state.plr_i_chamber, cosm.chambers.len()))
				}
			},

			Self::Set => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, Context{sys, cosm, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Set => {
				if ether.mode == Mode::Fly {
					let glob_ox = V3Df{x: 1.0, y: 0.0, z: 0.0};
					let glob_oy = V3Df{x: 0.0, y: 1.0, z: 0.0};
					let glob_oz = V3Df{x: 0.0, y: 0.0, z: 1.0};

					// Position
					let mut d_pos = V3Df{x: 0.0, y: 0.0, z: 0.0};

					let mut jump: bool = false;

					if sys.poll_key(Key::W) { d_pos.x += 1.0; } // forward
					if sys.poll_key(Key::S) { d_pos.x -= 1.0; } // backward
					if sys.poll_key(Key::A) { d_pos.y += 1.0; } // left
					if sys.poll_key(Key::D) { d_pos.y -= 1.0; } // right
					if sys.poll_key(Key::Q) { d_pos.z += 1.0; } // up
					if sys.poll_key(Key::Z) { d_pos.z -= 1.0; } // down

					if sys.poll_key(Key::Space) {
						d_pos = V3Df{x: 1.0, y: 0.0, z: 0.0}; // forward
						jump = true;
					}

					if !d_pos.is_zero() {
						let d_pos = d_pos.rotated3(&glob_ox, state.plr_att.ori.roll, &glob_oy, state.plr_att.ori.pitch, &glob_oz, state.plr_att.ori.yaw);
						let d_pos = V3Di{x: (d_pos.x * 65536.0) as i64, y: (d_pos.y * 65536.0) as i64, z: (d_pos.z * 65536.0) as i64};
						let ray_dist = cosm.ray_dist(state.plr_i_chamber, &state.plr_att.pos, &d_pos); // distance to wall in the direction of movement

						let max_step_distance =  (ray_dist - (SHORT_STEP_DISTANCE >> 1)).max(0); // to wall, with small gap

						let step_distance = match jump {
							true => max_step_distance,
							false => match sys.test_key(Key::LShift) || sys.test_key(Key::RShift) {
								true => LONG_STEP_DISTANCE.min(max_step_distance),
								false => SHORT_STEP_DISTANCE.min(max_step_distance)
							}
						};

						if ray_dist > step_distance { // no obstacle
							let n_pos = state.plr_att.pos.added(&(d_pos.scaled(step_distance)));
							// Why not test only current & adjacent chambers?
							// Because for now, player's move can be long enough to "jump over"
							// "thin" adjacent chamber(s) to some other one...
							if let Some(i) = cosm.chamber_by_pos(&n_pos) {
								state.plr_att.pos = n_pos;
								state.plr_i_chamber = i;
							}
						}
					}

					// Orientation
					let d_mouse_hor_angle = ROTATION_MOUSE_SENSITIVITY_MULT * (sys.mouse_dx() as f64) * PI / 180.0;
					let d_mouse_ver_angle = ROTATION_MOUSE_SENSITIVITY_MULT * (sys.mouse_dy() as f64) * PI / 180.0;
					let (d_yaw, d_pitch, d_roll) = match (sys.test_btn(Button::Left), sys.test_btn(Button::Middle), sys.test_btn(Button::Right)) {
						(false, false, false) => (-d_mouse_hor_angle, d_mouse_ver_angle, 0.0f64),
						(false, true, false) => (0.0f64, 0.0f64, d_mouse_hor_angle),
						_ => (0.0f64, 0.0f64, 0.0f64)
					};

					// TODO: wobbling, if any, should be independent of framerate...
					// let d_roll = d_roll + 0.02 * (0.125 * ((sys.now() as f64) / 1e6) * 2.0 * PI).sin() * PI / 180.0;

					if (d_yaw != 0.0) || (d_pitch != 0.0) || (d_roll != 0.0) {
						let (local_ox, local_oy, local_oz) = (
							glob_ox.rotated3(&glob_ox, state.plr_att.ori.roll, &glob_oy, state.plr_att.ori.pitch, &glob_oz, state.plr_att.ori.yaw),
							glob_oy.rotated3(&glob_ox, state.plr_att.ori.roll, &glob_oy, state.plr_att.ori.pitch, &glob_oz, state.plr_att.ori.yaw),
							glob_oz.rotated3(&glob_ox, state.plr_att.ori.roll, &glob_oy, state.plr_att.ori.pitch, &glob_oz, state.plr_att.ori.yaw)
						);

						let (ox, oy, oz) = (
							local_ox.rotated3(&local_ox, d_roll, &local_oy, d_pitch, &local_oz, d_yaw),
							local_oy.rotated3(&local_ox, d_roll, &local_oy, d_pitch, &local_oz, d_yaw),
							local_oz.rotated3(&local_ox, d_roll, &local_oy, d_pitch, &local_oz, d_yaw)
						);

						state.plr_att.ori = Orientation::from_reframe_axes(&ox, &oy, &oz);
					}

				}
				Ok(())
			},

			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

}
