/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use std::{
	collections::HashMap,
	fs
};

use crate::{
	base::{
		BASE_DIR,
		RAYNGIN_COPYR,
		RAYNGIN_TITLE,
		RAYNGIN_URL,
		PLAY_COPYR,
		PLAY_TITLE,
		PLAY_TITLE_EFFECT,
		PLAY_URL,
		SAVES_DIR,
		VERSION,
		RGB,
		PrependErrorString
	},
	sys::{
		Image,
		Key,
		System
	},
	cosm::CFont,
	state::State
};

use super::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	ERR_NOT_SET_CANNOT_FINISH,
	Context,
	Master
};

const MID: &str = "Meta";

const BACKGROUNDS_DIR: &str = "images/meta_backgrounds";
const BACKGROUND_SHADE: u8 = 0xF8;

const TEXT_COLOR: RGB = RGB{r: 0, g: 0xE8, b: 0};

const MENU_ITEM_ID_PREFIX: &str = "meta menu";
const MENU_ITEMS: [&str; 7] = ["resume", "help", "new", "load", "save", "about", "quit"]; // id-s for Lingua
const HELP_ID_PREFIX: &str = "meta help";
const STATE_LOSS_CONFIRM_ID: &str = "meta state-loss";
const ABOUT_ID_PREFIX: &str = "meta about";

const SAVE_INDEX_DIGITS: i32 = 4; // 0000 to 9999
const SAVE_NAME_FILENAME: &str = "name.txt";
const SAVE_STATE_FILENAME: &str = "state.jzn";
const MAX_SAVE_NAME_NCHARS: usize = 0x80;


#[derive(Clone, Copy, Eq, PartialEq)]
pub enum Screen {
	Menu,
	Help,
	New,
	Load,
	Save,
	About,
	StateLossConfirm
}

#[derive(Clone, Copy, Eq, PartialEq)]
enum Align {
	Left,
	Center,
	Right
}

pub struct Terminal {
	ch_height: i32,
	ncols: i32,
	nrows: i32,
	x_padding: i32,
	y_padding: i32,
	buf: Vec<char>
}

pub enum MMeta {
	Unset,
	Set {
		term: Terminal,

		scr: Screen,

		max_menu_item_name_nchars: i32,
		sel_menu_item: String,

		confirm_caller_scr: Screen,
		confirm_sel_yes: Option<bool>,

		savename_hmap: HashMap<i32, String>, // index to name
		i_saveslot: i32,
		max_save_index: i32,
		typing: bool, // true when typing save name
		typed_name: String,

		background: Option<Image>,
	}
}

fn load_rand_background(Context{sys, ..}: &mut Context) -> Option<Image> {
	let mut filenames = Vec::<String>::new();
	if let Ok(entry_iter) = fs::read_dir(format!("{}/{}", BASE_DIR, BACKGROUNDS_DIR)) {
		for entry in entry_iter {
			if let Ok(entry) = entry {
				if let Ok(e_type) = entry.file_type() {
					if e_type.is_file() {
						let filename = entry.file_name().into_string();
						if let Ok(filename) = filename {
							filenames.push(filename); // all files in that dir should be images, loadable with image crate
						}
					}
				}
			}
		}
	}

	match filenames.len() {
		0 => None,
		_ => {
			let i = sys.random::<usize>() % filenames.len();
			let img = Image::load(format!("{0}/{1}", BACKGROUNDS_DIR, & filenames[i]));
			match img {
				Ok(img) => {
					let (scale_x, scale_y) = ((sys.width() as f64) / (img.width() as f64), (sys.height() as f64) / (img.height() as f64));
					Some(img.scaled(scale_x.max(scale_y)))
				}
				Err(_) => None
			}
		}
	}
}

fn hashmap_saves() -> Result<HashMap<i32, String>, String> {
	let mut hmap = HashMap::<i32, String>::new();
	for entry in fs::read_dir(SAVES_DIR).pre_err("cannot read saves dir")? {
		let entry = entry.pre_err("cannot read saves dir entry")?;
		let e_type = entry.file_type().pre_err("cannot get file type of saves dir entry")?;
		if e_type.is_dir() {
			let dirname = entry.file_name().into_string().map_err(|_| String::from("cannot stringify name of saves dir subdir"))?;
			if dirname.chars().count() != (SAVE_INDEX_DIGITS as usize) {
				return Err(format!("'{}' does not consist of {} chars", &dirname, SAVE_INDEX_DIGITS));
			}
			let mut ind: i32 = 0;
			for (i, ch) in dirname.chars().enumerate() {
				match ch.to_digit(10) {
					Some(d) => {
						ind = ind * 10 + (d as i32);
					},
					None => return Err(format!("char {} of '{}' is not a digit", i + 1, &dirname))
				}
			}
			let save_name_filepath = format!("{}/{}/{}", SAVES_DIR, &dirname, SAVE_NAME_FILENAME);
			let mut name = String::from_utf8(fs::read(&save_name_filepath).pre_err(format!("cannot read '{}'", &save_name_filepath))?).pre_err(format!("cannot stringify '{}' content", &save_name_filepath))?;
			while name.chars().count() > MAX_SAVE_NAME_NCHARS { // TODO: truncate faster
				name.pop();
			}
			hmap.insert(ind, name);
		}
	}
	Ok(hmap)
}

impl Terminal {
	fn init(Context{sys, fonts, ..}: &mut Context) -> Result<Self, String> {
		let whitespace_img = System::make_text_image(fonts.get(CFont::Meta), " ", 0, TEXT_COLOR).pre_err("cannot render whitespace char of Meta font to calculate terminal dimensions")?;

		let (ch_width, ch_height) = (whitespace_img.width(), whitespace_img.height());
		let (ncols, nrows) = (sys.width() / ch_width, sys.height() / ch_height);
		let (x_padding, y_padding) = ((sys.width() - ncols * ch_width) >> 1, (sys.height() - nrows * ch_height) >> 1);

		let buf: Vec<char> = vec![' '; (ncols * nrows) as usize];

		Ok(Self{ch_height, ncols, nrows, x_padding, y_padding, buf})
	}

	fn show(&mut self, Context{sys, ref fonts, ..}: &mut Context) {
		if !sys.draw_locked() {
			let mut offs: usize = 0;
			for ty in 0..self.nrows {
				let mut strow = String::new();
				for _tx in 0..self.ncols {
					strow.push(self.buf[offs]);
					offs += 1;
				}
				let _ = sys.draw_text(fonts.get(CFont::Meta), strow, 0, TEXT_COLOR, 0, 0, - (sys.width() >> 1) + self.x_padding, - (sys.height() >> 1) + self.y_padding + self.ch_height * ty);
			}
		}
	}

	fn print(&mut self, text: impl ToString, x: i32, y: i32) {
		if (y >= 0) && (y < self.nrows) {
			let text = text.to_string();
			for (i, ch) in text.chars().enumerate() {
				let x = x + (i as i32);
				if (x >= 0) && (x < self.ncols) {
					self.buf[(y * self.ncols + x) as usize] = ch;
				}
			}
		}
	}

	fn print_align(&mut self, text: impl ToString, y: i32, align: Align) {
		let text_nchars = text.to_string().chars().count() as i32;
		let x = match align {
			Align::Left => 0,
			Align::Center => (self.ncols - text_nchars) >> 1,
			Align::Right => self.ncols - text_nchars
		};
		self.print(text, x, y);
	}

	fn print_fill_rect(&mut self, x1: i32, y1: i32, x2: i32, y2: i32, fill: char) {
		let (x1, y1, x2, y2) = (x1.min(x2), y1.min(y2), x1.max(x2), y1.max(y2));
		let fill = format!("{}", fill);
		for y in y1..=y2 {
			for x in x1..=x2 {
				self.print(&fill, x, y);
			}
		}
	}

	fn print_frame(&mut self, x1: i32, y1: i32, x2: i32, y2: i32) {
		let (x1, y1, x2, y2) = (x1.min(x2), y1.min(y2), x1.max(x2), y1.max(y2));
		self.print("┌", x1, y1); self.print("┐", x2, y1);
		self.print("└", x1, y2); self.print("┘", x2, y2);
		for x in (x1+1)..x2 {
			self.print("─", x, y1); self.print("─", x, y2);
		}
		for y in (y1+1)..y2 {
			self.print("│", x1, y); self.print("│", x2, y);
		}
	}

	fn clear(&mut self) {
		for ch in &mut self.buf {
			*ch = ' ';
		}
	}

}

impl MMeta {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}

	fn pause_reset(&mut self, Context{sys, ether, ..}: &mut Context) {
		ether.paused = true;
		sys.pause_audio();

		if let Self::Set{ref mut scr, ref mut sel_menu_item, ..} = *self {
			*scr = Screen::Menu;
			*sel_menu_item = String::from("resume");
		}
	}

	fn menu(&mut self, Context{sys, lingua, ether, ..}: &mut Context) {
		if let Self::Set{ref mut term, ref mut scr, max_menu_item_name_nchars, ref mut sel_menu_item, ref mut confirm_sel_yes, ..} = *self {
			// Print menu
			let mut i_sel = 0;
			for (i, item) in MENU_ITEMS.iter().enumerate() {
				let (i, item) = (i as i32, item.to_string());
				let item_name = lingua.get_or_echo(format!("{} {}", MENU_ITEM_ID_PREFIX, &item));
				let x = term.ncols >> 1;
				let y = (term.nrows >> 1) + 2 * (i - ((MENU_ITEMS.len() as i32) >> 1));
				term.print_align(item_name, y, Align::Center);
				if item.eq(sel_menu_item) { // framed
					i_sel = i;
					term.print_frame(x - (max_menu_item_name_nchars >> 1) - 3, y - 1, x + (max_menu_item_name_nchars >> 1) + 3, y + 1);
				}
			}

			// Print version
			term.print_align(format!("v{}", VERSION), term.nrows - 1, Align::Left);

			// Print "enshaped" info
			term.print_align(format!("Enshaped with {}", RAYNGIN_TITLE), term.nrows - 1, Align::Right);

			// Print title
			term.print_align(String::from(PLAY_TITLE_EFFECT), 1, Align::Center);

			// Process input

			// Change selected menu item
			let mut d_i_sel: i32 = 0;
			if sys.poll_keys(&[Key::Up, Key::Kp8]) {
				d_i_sel = -1;
			} else if sys.poll_keys(&[Key::Down, Key::Kp2]) {
				d_i_sel = 1;
			} else if sys.poll_keys(&[Key::Home, Key::Kp7, Key::PageUp, Key::Kp9]) {
				d_i_sel = - (MENU_ITEMS.len() as i32);
			} else if sys.poll_keys(&[Key::End, Key::Kp1, Key::PageDown, Key::Kp3]) {
				d_i_sel = MENU_ITEMS.len() as i32;
			}
			if d_i_sel != 0 {
				i_sel = (i_sel + d_i_sel).max(0).min((MENU_ITEMS.len() as i32) - 1);
				*sel_menu_item = MENU_ITEMS[i_sel as usize].to_string();
			}

			// Select menu item
			if sys.poll_keys(&[Key::Enter, Key::KpEnter]) {
				match sel_menu_item.as_str() {
					"resume" => {
						ether.meta = false;
					},
					"help" => {
						*scr = Screen::Help;
					},
					"new" => {
						*confirm_sel_yes = None; // state loss must be confirmed
						*scr = Screen::New;
					},
					"load" => {
						*confirm_sel_yes = None; // state loss must be confirmed
						*scr = Screen::Load;
					},
					"save" => {
						*scr = Screen::Save;
					},
					"about" => {
						*scr = Screen::About;
					},
					"quit" => {
						ether.quit = true;
					}
					_ => {}
				}
			} else if sys.poll_key(Key::Escape) {
				ether.meta = false;
			}
		}
	}

	fn help(&mut self, Context{sys, lingua, ..}: &mut Context) {
		if let Self::Set{ref mut term, ref mut scr, ..} = *self {
			// Print help title
			term.print_align(lingua.get_or_echo(format!("{} title", HELP_ID_PREFIX)), 0, Align::Center);

			// Print help strings
			let mut i: i32 = 1;
			while let Some(text) = lingua.get(format!("{} {:02}", HELP_ID_PREFIX, i)) {
				term.print_align(text, 1 + i, Align::Left);
				i += 1;
			}

			// Process input

			// Return to menu
			if sys.poll_key(Key::Escape) {
				*scr = Screen::Menu;
			}
		}
	}

	fn new_play(&mut self, Context{state, ether, ..}: &mut Context) {
		if let Self::Set{ref mut scr, ref mut confirm_caller_scr, ref mut confirm_sel_yes, ..} = *self {
			match *confirm_sel_yes {
				Some(true) => {
					**state = State::new(); // hmm, double deref
					ether.replay = true;
				},
				Some(false) => {
					*scr = Screen::Menu;
					*confirm_sel_yes = None;
				}
				None => { // so, this "screen" just switches to confirmation one
					*scr = Screen::StateLossConfirm;
					*confirm_caller_scr = Screen::New;
					*confirm_sel_yes = Some(false); // do not lose state by default
				}
			}
		}
	}

	fn save_load(&mut self, Context{sys, lingua, state, ether, ..}: &mut Context) -> Result<(), String> {
		if let Self::Set{ref mut term, ref mut scr, ref mut confirm_caller_scr, ref mut confirm_sel_yes, ref mut savename_hmap, ref mut i_saveslot, max_save_index, ref mut typing, ref mut typed_name, ..} = *self {
			let load = *scr == Screen::Load;

			match (load, *confirm_sel_yes) {
				(true, Some(true)) => {
					let s_fpath = format!("{0:01$}/{2}", *i_saveslot, SAVE_INDEX_DIGITS as usize, SAVE_STATE_FILENAME);
					**state = State::load(&s_fpath).pre_err(format!("cannot load state from '{}' in saves dir", &s_fpath))?;
					ether.replay = true;
				},
				(true, Some(false)) => {
					*confirm_sel_yes = None;
				},
				_ => {
					// Print screen title
					let title_text = lingua.get_or_echo(match load {true => "meta load title", false => "meta save title"});
					term.print_align(title_text, 0, Align::Center);

					let half_display_save_slots = ((term.nrows - 6) >> 2).max(0);

					// Print save names
					for i in (-half_display_save_slots)..=half_display_save_slots { // "screen" slot index
						let y = (term.nrows >> 1) + (i << 1);
						let j = i + (*i_saveslot); // "absolute" slot index
						if (j >= 0) && (j <= max_save_index) {
							term.print(format!("{0:01$} • ", j, SAVE_INDEX_DIGITS as usize), 2, y);
							if (i == 0) && !load && *typing {
								term.print(format!("{0:.1$}█", typed_name, (term.ncols - 7 - SAVE_INDEX_DIGITS) as usize), 5 + SAVE_INDEX_DIGITS, y);
							} else {
								match savename_hmap.get(&j) {
									Some(name) => {
										term.print(format!("{0:.1$}", name, (term.ncols - 7 - SAVE_INDEX_DIGITS) as usize), 5 + SAVE_INDEX_DIGITS, y);
									},
									None => term.print_fill_rect(5 + SAVE_INDEX_DIGITS, y, term.ncols - 3, y, '░')
								}
							}
						}
					}

					// Print frame around selected save slot (central)
					term.print_frame(0, (term.nrows >> 1) - 1, term.ncols - 1, (term.nrows >> 1) + 1);

					// Print hints
					if *typing {
						term.print_align(lingua.get_or_echo("meta save typing hint act"), term.nrows - 1, Align::Left);
						term.print_align(lingua.get_or_echo("meta save typing hint del"), term.nrows - 1, Align::Right);
					} else {
						term.print_align(lingua.get_or_echo("meta saveload hint act"), term.nrows - 1, Align::Left);
						term.print_align(lingua.get_or_echo("meta saveload hint del"), term.nrows - 1, Align::Right);
					}

					// Process input

					if *typing {
						if sys.poll_key(Key::Escape) {
							*typing = false;
						} else if sys.poll_keys(&[Key::Enter, Key::KpEnter]) {
							*typing = false;
							if !typed_name.is_empty() {
								// Update hashmap
								savename_hmap.insert(*i_saveslot, typed_name.clone());
								// Create subdir in saves dir if needed
								let s_dpath = format!("{0:01$}", *i_saveslot, SAVE_INDEX_DIGITS as usize);
								fs::create_dir_all(format!("{}/{}", SAVES_DIR, &s_dpath)).pre_err(format!("cannot create '{}' subdir in saves dir", &s_dpath))?;
								// Write save name
								let s_fpath = format!("{}/{}", &s_dpath, SAVE_NAME_FILENAME);
								fs::write(format!("{}/{}", SAVES_DIR, &s_fpath), typed_name.as_bytes()).pre_err(format!("cannot write save name to '{}' in saves dir", &s_fpath))?;
								// Write state
								let s_fpath = format!("{}/{}", &s_dpath, SAVE_STATE_FILENAME);
								state.save(&s_fpath).pre_err(format!("cannot save state to '{}' in saves dir", &s_fpath))?;
							}
						} else if sys.poll_keys(&[Key::Delete, Key::KpPeriod]) {
							typed_name.clear();
						} else if sys.poll_key(Key::Backspace) {
							typed_name.pop();
						} else if let Some(ch) = sys.get_poll_char() {
							if typed_name.chars().count() < MAX_SAVE_NAME_NCHARS {
								typed_name.push(ch);
							}
						}

					} else { // not typing
						if sys.poll_keys(&[Key::Enter, Key::KpEnter]) {
							if load {
								if let Some(_) = savename_hmap.get(i_saveslot) { // there is something to load
									// Switch to confirmation screen
									*scr = Screen::StateLossConfirm;
									*confirm_caller_scr = Screen::Load;
									*confirm_sel_yes = Some(false); // do not lose state by default
								}
							} else { // start typing save name
								*typing = true;
								*typed_name = match savename_hmap.get(i_saveslot) {
									Some(name) => name.clone(),
									None => String::new()
								};
							}
						} else if sys.poll_keys(&[Key::Up, Key::Kp8]) {
							*i_saveslot = ((*i_saveslot) - 1).max(0).min(max_save_index);
						} else if sys.poll_keys(&[Key::Down, Key::Kp2]) {
							*i_saveslot = ((*i_saveslot) + 1).max(0).min(max_save_index);
						} else if sys.poll_keys(&[Key::PageUp, Key::Kp9]) {
							*i_saveslot = ((*i_saveslot) - ((half_display_save_slots << 1) + 1)).max(0).min(max_save_index);
						} else if sys.poll_keys(&[Key::PageDown, Key::Kp3]) {
							*i_saveslot = ((*i_saveslot) + ((half_display_save_slots << 1) + 1)).max(0).min(max_save_index);
						} else if sys.poll_keys(&[Key::Home, Key::Kp7]) {
							*i_saveslot = 0;
						} else if sys.poll_keys(&[Key::End, Key::Kp1]) {
							*i_saveslot = max_save_index;
						} else if sys.poll_keys(&[Key::Delete, Key::KpPeriod]) && sys.test_keys(&[Key::LShift, Key::RShift]) {
							if let Some(_) = savename_hmap.get(i_saveslot) {
								savename_hmap.remove(i_saveslot);
								let s_dpath = format!("{0:01$}", *i_saveslot, SAVE_INDEX_DIGITS as usize);
								fs::remove_dir_all(format!("{}/{}", SAVES_DIR, &s_dpath)).pre_err(format!("cannot remove '{}' subdir from saves dir", &s_dpath))?;
							}
						} else if sys.poll_key(Key::Escape) {
							*scr = Screen::Menu;
						}
					}
				}
			}
		}

		Ok(())
	}

	fn about(&mut self, Context{sys, lingua, ..}: &mut Context) {
		if let Self::Set{ref mut term, ref mut scr, ..} = *self {
			// Print title
			term.print_align(lingua.get_or_echo(format!("{} title", ABOUT_ID_PREFIX)), 0, Align::Center);

			// Print about strings
			term.print_align(String::from(RAYNGIN_TITLE), 3, Align::Center);
			term.print_align(format!("v{}", VERSION), 4, Align::Center);
			term.print_align(String::from(RAYNGIN_COPYR), 5, Align::Center);
			term.print_align(String::from(RAYNGIN_URL), 6, Align::Center);

			term.print_align(String::from(PLAY_TITLE), 9, Align::Center);
			term.print_align(format!("v{}", VERSION), 10, Align::Center);
			term.print_align(String::from(PLAY_COPYR), 11, Align::Center);
			term.print_align(String::from(PLAY_URL), 12, Align::Center);

			term.print_frame(0, 14, term.ncols - 1, 14);

			// Print license
			let mut i = 1;
			while let Some(text) = lingua.get(format!("{} gpl {:02}", ABOUT_ID_PREFIX, i)) {
				term.print_align(text, 15 + i, Align::Left);
				i += 1;
			}

			// Process input

			// Return to menu
			if sys.poll_key(Key::Escape) {
				*scr = Screen::Menu;
			}
		}
	}

	fn state_loss_confirm(&mut self, Context{sys, lingua, ..}: &mut Context) {
		if let Self::Set{ref mut term, ref mut scr, ref mut confirm_sel_yes, confirm_caller_scr, ..} = *self {
			// Print title
			term.print_align(lingua.get_or_echo(format!("{} warning", STATE_LOSS_CONFIRM_ID)), (term.nrows >> 1) - 4, Align::Center);

			// Print warning context
			term.print_align(lingua.get_or_echo(format!("{} description", STATE_LOSS_CONFIRM_ID)), (term.nrows >> 1) - 2, Align::Center);
			term.print_align(lingua.get_or_echo(format!("{} question", STATE_LOSS_CONFIRM_ID)), (term.nrows >> 1) - 1, Align::Center);

			// Print "yes" and "no" items
			let (yes_text, no_text) = (
				lingua.get_or_echo(format!("{} answer yes", STATE_LOSS_CONFIRM_ID)),
				lingua.get_or_echo(format!("{} answer no", STATE_LOSS_CONFIRM_ID))
			);
			term.print_align(&yes_text, (term.nrows >> 1) + 2, Align::Center);
			term.print_align(&no_text, (term.nrows >> 1) + 4, Align::Center);

			let max_text_nchars = (yes_text.chars().count() as i32).max(no_text.chars().count() as i32);

			let rect_y: i32 = (term.nrows >> 1) + match *confirm_sel_yes {
				Some(false) => 4, Some(true) => 2, _ => 6
			};

			// Print frame around selected item
			term.print_frame(((term.ncols - max_text_nchars) >> 1) - 3, rect_y - 1, ((term.ncols + max_text_nchars) >> 1) + 3, rect_y + 1);

			// Process input
			if sys.poll_keys(&[Key::Up, Key::Kp8, Key::PageUp, Key::Kp9, Key::Home, Key::Kp7]) {
				*confirm_sel_yes = Some(true);
			} else if sys.poll_keys(&[Key::Down, Key::Kp2, Key::PageDown, Key::Kp3, Key::End, Key::Kp1]) {
				*confirm_sel_yes = Some(false);
			} else if sys.poll_keys(&[Key::Enter, Key::KpEnter]) {
				*scr = confirm_caller_scr;
			}
		}
	}

}

impl Master for MMeta {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, ctx: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				ctx.lingua.load(&["meta"])?;

				let term = Terminal::init(ctx)?;

				let scr = Screen::Menu;

				let max_menu_item_name_nchars = MENU_ITEMS.iter().fold(0, |max_nchars, s| max_nchars.max(ctx.lingua.get_or_echo(format!("{} {}", MENU_ITEM_ID_PREFIX, s.to_string())).chars().count())) as i32;
				let sel_menu_item = String::from("resume");

				let confirm_caller_scr = Screen::New;
				let confirm_sel_yes: Option<bool> = None;

				let savename_hmap = hashmap_saves().pre_err(format!("cannot hashmap saves from '{}'", SAVES_DIR))?;
				let i_saveslot: i32 = 0;
				let mut max_save_index: i32 = 1;
				for _i in 0..SAVE_INDEX_DIGITS {
					max_save_index *= 10;
				}
				max_save_index -= 1;
				let typing = false;
				let typed_name = String::new();

				let background = load_rand_background(ctx);

				*self = Self::Set {
					term,
					scr,
					max_menu_item_name_nchars, sel_menu_item,
					confirm_caller_scr, confirm_sel_yes,
					savename_hmap, i_saveslot, max_save_index, typing, typed_name,
					background
				};
				Ok(())
			},

			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, ctx: &mut Context) -> Result<(), String> {
		if let Self::Unset = self {
			return Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN));
		}

		if !ctx.ether.paused { // pause play and reset menu
			self.pause_reset(ctx);
		}

		if let Self::Set{ref mut term, ..} = *self {
			term.clear();
		}

		let scr = match *self {
			Self::Set{scr, ..} => scr,
			_ => Screen::Menu
		};

		match scr {
			Screen::Menu => self.menu(ctx),
			Screen::Help => self.help(ctx),
			Screen::New => self.new_play(ctx),
			Screen::Load => self.save_load(ctx).pre_err("cannot load")?,
			Screen::Save => self.save_load(ctx).pre_err("cannot save")?,
			Screen::About => self.about(ctx),
			Screen::StateLossConfirm => self.state_loss_confirm(ctx)
		};

		// Draw
		if let Self::Set{ref background, ref mut term, ..} = *self {
			if let Some(img) = background {
				let _ = ctx.sys.draw_image(img, - (img.width() >> 1), - (img.height() >> 1));
				let _ = ctx.sys.shade(BACKGROUND_SHADE);
			} else {
				let _ = ctx.sys.shade(0xFF); // black background
			}

			term.show(ctx);
		}

		// DEBUG
		// let _ = ctx.sys.draw_text(ctx.fonts.get(CFont::System), format!("FPS = {:.3}", ctx.sys.fps()), 0, RGB{r: 0xFF, g: 0xFF, b: 0xFF}, 0xFF, 0, -(ctx.sys.width()>>1), -(ctx.sys.height()>>1));

		ctx.sys.sync()?;

		Ok(())
    }

	fn finish(&mut self, _ctx: &mut Context) -> Result<(), String> {
		match *self {
			Self::Set{..} => {
				Ok(())
			},
			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_FINISH))
		}
	}

}
