/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use crate::{
	base::{
		ARGB,
		RGB
	},
	sys::{
		Button,
		Key
	},
	cosm::CFont
};

use super::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	Context,
	Master,
	Mode
};

const MID: &str = "Inventory";

const SHADE: u8 = 0xE0;
const PADDING: i32 = 8;

const ITEM_ID_PREFIX: &str = "item";
const ITEM_TEXT_COLOR: RGB = RGB{r: 0xA0, g: 0xA0, b: 0x90};
const ITEM_TEXT_SEL_COLOR: RGB = RGB{r: 0xF0, g: 0xF0, b: 0xD8};

const COMBINE_FAILED_DESCRIPTIONS_NUM: u32 = 4;

pub enum MInventory {
	Unset,
	Set {
		target_item: String
	}
}

impl MInventory {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}
}

impl Master for MInventory {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, Context{lingua, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				lingua.load(&["items"])?;
				let target_item = String::new();
				*self = Self::Set{target_item};
				Ok(())
			},
			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, Context{sys, ref fonts, lingua, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
		 	Self::Set{ref mut target_item} => {
				// Draw if drawing unlocked

				if !sys.draw_locked() {
					// List of items
					if (ether.mode == Mode::Inventory) || (ether.mode == Mode::InventoryCombine) || ((ether.mode == Mode::Descript) && (ether.pre_descript_mode == Mode::Inventory)) {
						let _ = sys.draw_rect(-(sys.width()>>1), -(sys.height()>>1), (sys.width()>>1)-1, (sys.height()>>1)-1, ARGB{a: SHADE, r: 0, g: 0, b: 0}, true); // shade view
						let fh = fonts.get(CFont::Inventory).height();
						let mut y = (sys.height() >> 1) - fh * (state.items.len() as i32) - PADDING;
						if ether.mode != Mode::InventoryCombine { // display names of items
							for item in &state.items {
								let color = match state.sel_item.eq(item) {
									true => ITEM_TEXT_SEL_COLOR,
									false => ITEM_TEXT_COLOR
								};
								let txt = lingua.get_or_echo(format!("{} {}", ITEM_ID_PREFIX, item));
								let _ = sys.draw_text(fonts.get(CFont::Inventory), txt, 0, color, 0, 0, -(sys.width()>>1) + PADDING, y);
								y += fh;
							}
						} else { // display "item name" or "target item name + selected item name"
							y += fh; // omit selected item in list (it cannot be combined with itself)
							for item in &state.items {
								if !state.sel_item.eq(item) {
									let (txt, color) = match (*target_item).eq(item) {
										true => (format!("{} + {}", lingua.get_or_echo(format!("{} {}", ITEM_ID_PREFIX, item)), lingua.get_or_echo(format!("{} {}", ITEM_ID_PREFIX, &state.sel_item))), ITEM_TEXT_SEL_COLOR),
										false => (lingua.get_or_echo(format!("{} {}", ITEM_ID_PREFIX, item)), ITEM_TEXT_COLOR)
									};
									let _ = sys.draw_text(fonts.get(CFont::Inventory), txt, 0, color, 0, 0, -(sys.width()>>1) + PADDING, y);
									y += fh;
								}
							}
						}
					}

				}

				// Process input regardless of drawing lock

				match ether.mode {
					Mode::Fly => {
						// Switch to Inventory mode
						if !sys.test_modkeys() && !sys.test_btn(Button::Middle) && (sys.mouse_dw() != 0) && (state.items.len() > 0) {
							// Determine index of selected item
							let mut i = match state.sel_item_ind() {
								Some(i) => i as i32,
								None => 0
							};
							i = (i - sys.mouse_dw()).max(0).min((state.items.len() - 1) as i32);
							state.sel_item = state.items[i as usize].clone();
							ether.mode = Mode::Inventory;
						}
					},

					Mode::Inventory => {
						if !sys.test_modkeys() && (sys.mouse_dw() != 0) { // change selected item
							// Determine index of selected item
							let mut i = match state.sel_item_ind() {
								Some(i) => i as i32,
								None => 0
							};
							i = (i - sys.mouse_dw()).max(0).min((state.items.len() - 1) as i32);
							state.sel_item = state.items[i as usize].clone();
						} else if sys.poll_btn(Button::Right) { // show item descriptions
							ether.describe(state.sel_item.clone());
						} else if sys.poll_btn(Button::Left) { // exit Inventory with selected item
							ether.mode = Mode::Fly;
						} else if (sys.mouse_dw() == 0) && sys.poll_btn(Button::Middle) { // select no item
							state.sel_item.clear();
							ether.mode = Mode::Fly;
						} else if sys.test_keys(&[Key::LShift, Key::RShift]) && !sys.test_keys(&[Key::LAlt, Key::RAlt, Key::LCtrl, Key::RCtrl]) && (state.items.len() > 1) {
							let i = if state.sel_item.eq(&state.items[0]) { 1 } else { 0 };
							*target_item = state.items[i].clone(); // 1st not selected item
							ether.mode = Mode::InventoryCombine;
						}
					},

					Mode::InventoryCombine => {
						if sys.test_keys(&[Key::LShift, Key::RShift]) && !sys.test_keys(&[Key::LAlt, Key::RAlt, Key::LCtrl, Key::RCtrl]) && (sys.mouse_dw() != 0) { // change target item
							let i_s = match state.sel_item_ind() {
								Some(i) => i as i32,
								None => 0
							};
							let mut i_t = match state.item_ind(&target_item) {
								Some(i) => i as i32,
								None => 0
							};
							if i_t >= i_s {
								i_t -= 1; // index in list without sel_item
							}
							i_t = (i_t - sys.mouse_dw()).max(0).min((state.items.len() - 2) as i32);
							if i_t >= i_s {
								i_t += 1; // index in list with sel_item
							}
							*target_item = state.items[i_t as usize].clone();
						} else if !sys.test_keys(&[Key::LShift, Key::RShift]) { // attempt to combine selected and target item
							let mut combination = vec![state.sel_item.clone(), target_item.clone()];
							combination.sort(); // in alphabetical order
							let combination = format!("{} + {}", &(combination[0]), &(combination[1]));

							match combination.as_str() {
								"melody + memories" => {
									state.remove_item("memories")?;
									state.add_and_sel_item("memories-mus")?;
									ether.describe("melody + memories");
								},
								_ => ether.describe(format!("_ + _ {}", 1 + (sys.random::<u32>() % COMBINE_FAILED_DESCRIPTIONS_NUM))) // "random" (no)
							}; // after all descriptions, mode becomes Inventory
						}
					},

					_ => ()
				}

				Ok(())
			},

			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

}
