/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use crate::{
	base::{
		ARGB,
		RGB
	},
	sys::{
		Button,
		Image,
		Key,
		System
	},
	cosm::CFont
};

use super::{
	ERR_ALREADY_SET,
	ERR_NOT_SET_CANNOT_RUN,
	Context,
	Master,
	Mode
};

const MID: &str = "HUD";

const TEMPUS_START: i128 = 4_599_500_746_867_056_341_626_445_612_537_592i128;
const TEMPUS_TEXT_COLOR: RGB = RGB{r: 0x80, g: 0xE0, b: 0x80};
const TEMPUS_TEXT_SHADE: u8 = 0xC0;
const TEMPUS_TEXT_MARGIN: i32 = 1;

const REACHABLE_ID_PREFIX: &str = "ent";
const REACHABLE_TEXT_COLOR: RGB = RGB{r: 0x90, g: 0x90, b: 0xA0};
const REACHABLE_TEXT_SEL_COLOR: RGB = RGB{r: 0xD8, g: 0xD8, b: 0xF0};
const REACHABLE_TEXT_SHADE: u8 = 0xE8;
const REACHABLE_TEXT_MARGIN: i32 = 1;
const REACHABLE_FRAME_INT_COLOR: ARGB = ARGB{a: 0xF0, r: 0x18, g: 0x18, b: 0x20};
const REACHABLE_FRAME_EXT_COLOR: ARGB = ARGB{a: 0xE0, r: 0x80, g: 0x80, b: 0xA0};
const REACHABLE_FRAME_SEL_INT_COLOR: ARGB = ARGB{a: 0xF0, r: 0x20, g: 0x20, b: 0x30};
const REACHABLE_FRAME_SEL_EXT_COLOR: ARGB = ARGB{a: 0xE0, r: 0xB0, g: 0xB0, b: 0xE0};
const REACHABLE_FRAME_THICKNESS_BINLOG: i32 = 3;
const REACHABLE_FRAME_THICKNESS: i32 = 1 << REACHABLE_FRAME_THICKNESS_BINLOG;

const ITEM_ID_PREFIX: &str = "item";
const ITEM_TEXT_COLOR: RGB = RGB{r: 0xF0, g: 0xF0, b: 0xD8};
const ITEM_TEXT_SHADE: u8 = 0xE8;
const ITEM_TEXT_MARGIN: i32 = 1;
const ITEM_FRAME_INT_COLOR: ARGB = ARGB{a: 0xF0, r: 0x20, g: 0x20, b: 0x18};
const ITEM_FRAME_EXT_COLOR: ARGB = ARGB{a: 0xE0, r: 0xA0, g: 0xA0, b: 0x80};
const ITEM_FRAME_THICKNESS_BINLOG: i32 = 3;
const ITEM_FRAME_THICKNESS: i32 = 1 << ITEM_FRAME_THICKNESS_BINLOG;

const DESCRIPTION_ID_PREFIX: &str = "desc";
const INVENTORY_DESCRIPTION_ID_PREFIX: &str = "inv-desc";
const PASSIVE_DESCRIPTION_ID_PREFIX: &str = "pas-desc";
const DESCRIPTION_TEXT_COLOR: RGB = RGB{r: 0xE0, g: 0xE0, b: 0xE0};
const DESCRIPTION_TEXT_SHADE: u8 = 0xD0;
const DESCRIPTION_TEXT_MARGIN: i32 = 2;
const DESCRIPTION_FRAME_INT_COLOR: ARGB = ARGB{a: 0xF0, r: 0x20, g: 0x20, b: 0x20};
const DESCRIPTION_FRAME_EXT_COLOR: ARGB = ARGB{a: 0xE0, r: 0xA0, g: 0xA0, b: 0xA0};
const DESCRIPTION_FRAME_THICKNESS_BINLOG: i32 = 3;

pub enum MHUD {
	Unset,
	Set {
		curr_desc_i: i32,
		curr_desc_txt: String,
		curr_desc_res_img: Result<Image, String> // "cache"
	}
}

impl MHUD {
	pub fn new() -> Box<Self> {
		Box::new(Self::Unset)
	}
}

impl Master for MHUD {
	fn id(&self) -> String {
		String::from(MID)
	}

	fn start(&mut self, Context{fonts, ..}: &mut Context) -> Result<(), String> {
		match *self {
			Self::Unset => {
				let curr_desc_i: i32 = 0;
				let curr_desc_txt = String::new();
				let curr_desc_res_img = System::make_text_image(fonts.get(CFont::Descript), &curr_desc_txt, 0, RGB{r: 0, g: 0, b: 0});
				*self = Self::Set{curr_desc_i, curr_desc_txt, curr_desc_res_img};
				Ok(())
			},
			Self::Set{..} => Err(format!("{} {}", MID, ERR_ALREADY_SET))
		}
	}

    fn run(&mut self, Context{sys, ref fonts, lingua, state, ether, ..}: &mut Context) -> Result<(), String> {
		match *self {
		 	Self::Set{ref mut curr_desc_i, ref mut curr_desc_txt, ref mut curr_desc_res_img} => {
				// Draw if drawing unlocked

				if !sys.draw_locked() {
					// Tempus
					match ether.mode { Mode::Fly | Mode::Passive | Mode::Inventory | Mode::InventoryCombine | Mode::Descript => {
						let txt = format!("{} a.D.", TEMPUS_START + state.time / 1000000);
						if let Ok(txt_img) = System::make_text_image(fonts.get(CFont::HUD_Tempus), &txt, 0, TEMPUS_TEXT_COLOR) {
							let _ = sys.draw_text_image(&txt_img, TEMPUS_TEXT_SHADE, TEMPUS_TEXT_MARGIN, -(txt_img.width() >> 1) - TEMPUS_TEXT_MARGIN, -(sys.height() >> 1));
						}
					}, _ => () }

					if (ether.mode == Mode::Fly) || ((ether.mode == Mode::Descript) && (ether.pre_descript_mode == Mode::Fly)) {
						// Reachables
						let fh = fonts.get(CFont::HUD_Reachable).height();
						let mut y = (sys.height() >> 1) - fh - REACHABLE_FRAME_THICKNESS - (REACHABLE_TEXT_MARGIN << 1);
						for r in &ether.reachables {
							let (text_color, frame_int_color, frame_ext_color) = match ether.sel_reachable.eq(r) {
								true => (REACHABLE_TEXT_SEL_COLOR, REACHABLE_FRAME_SEL_INT_COLOR, REACHABLE_FRAME_SEL_EXT_COLOR),
								false => (REACHABLE_TEXT_COLOR, REACHABLE_FRAME_INT_COLOR, REACHABLE_FRAME_EXT_COLOR)
							};
							let txt = lingua.get_or_echo(format!("{} {}", REACHABLE_ID_PREFIX, r));
							if let Ok(txt_img) = System::make_text_image(fonts.get(CFont::HUD_Reachable), &txt, 0, text_color) {
								let x = (sys.width() >> 1) - txt_img.width() - REACHABLE_FRAME_THICKNESS;
								let _ = sys.draw_text_image(&txt_img, REACHABLE_TEXT_SHADE, REACHABLE_TEXT_MARGIN, x, y);
								let _ = sys.draw_frame(x, y, txt_img.width() + (REACHABLE_TEXT_MARGIN << 1), fh + (REACHABLE_TEXT_MARGIN << 1), frame_int_color, frame_ext_color, REACHABLE_FRAME_THICKNESS_BINLOG);
							}
							y -= fh + (REACHABLE_FRAME_THICKNESS << 1) + (REACHABLE_TEXT_MARGIN << 1);
						}

						// Selected item from inventory
						if !state.sel_item.is_empty() {
							let txt = lingua.get_or_echo(format!("{} {}", ITEM_ID_PREFIX, &state.sel_item));
							if let Ok(txt_img) = System::make_text_image(fonts.get(CFont::Inventory), &txt, 0, ITEM_TEXT_COLOR) {
								let x = -(sys.width() >> 1) + ITEM_FRAME_THICKNESS;
								let y = (sys.height() >> 1) - ITEM_FRAME_THICKNESS - fonts.get(CFont::Inventory).height() - (ITEM_TEXT_MARGIN << 1);
								let _ = sys.draw_text_image(&txt_img, ITEM_TEXT_SHADE, ITEM_TEXT_MARGIN, x, y);
								let _ = sys.draw_frame(x, y, txt_img.width() + (ITEM_TEXT_MARGIN << 1), txt_img.height() + (ITEM_TEXT_MARGIN << 1), ITEM_FRAME_INT_COLOR, ITEM_FRAME_EXT_COLOR, ITEM_FRAME_THICKNESS_BINLOG);
							}
						}

					}

					// Show descriptions
					if ether.mode == Mode::Descript {
						// Check if it has just started
						*curr_desc_i = (*curr_desc_i).max(1);
						let desc_id_prefix = match ether.pre_descript_mode {
							Mode::Passive => PASSIVE_DESCRIPTION_ID_PREFIX,
							Mode::Inventory => INVENTORY_DESCRIPTION_ID_PREFIX,
							_ => DESCRIPTION_ID_PREFIX
						};
						match lingua.get(format!("{} {} {}", desc_id_prefix, &ether.descript, *curr_desc_i)) {
							Some(txt) => {
								// "Cache"
								if !txt.eq(curr_desc_txt) {
									*curr_desc_txt = txt;
									*curr_desc_res_img = System::make_text_image(fonts.get(CFont::Descript), curr_desc_txt, sys.width() >> 1, DESCRIPTION_TEXT_COLOR);
								}
								if let Ok(ref txt_img) = curr_desc_res_img {
									let (x, y) = (-(txt_img.width() >> 1) - DESCRIPTION_TEXT_MARGIN, -(txt_img.height() >> 1) - DESCRIPTION_TEXT_MARGIN);
									let _ = sys.draw_text_image(txt_img, DESCRIPTION_TEXT_SHADE, DESCRIPTION_TEXT_MARGIN, x, y);
									let _ = sys.draw_frame(x, y, txt_img.width() + (DESCRIPTION_TEXT_MARGIN << 1), txt_img.height() + (DESCRIPTION_TEXT_MARGIN << 1), DESCRIPTION_FRAME_INT_COLOR, DESCRIPTION_FRAME_EXT_COLOR, DESCRIPTION_FRAME_THICKNESS_BINLOG);
								}
							},
							None => { // no more descriptions
								*curr_desc_i = 0; // for next sequence of descriptions
								ether.mode = ether.pre_descript_mode; // Fly, Passive, Inventory, or?..
							}
						}
					}

				}

				// Process input regardless of drawing lock

				match ether.mode {
					Mode::Fly => {
						// Select reachable
						if (ether.reachables.len() > 0) && sys.test_keys(&[Key::LAlt, Key::RAlt]) && !sys.test_keys(&[Key::LCtrl, Key::RCtrl, Key::LShift, Key::RShift]) && (sys.mouse_dw() != 0) {
							// Determine index of selected reachable
							let mut i = match ether.sel_reachable_ind() {
								Some(i) => i as i32,
								None => 0
							};
							i = (i + sys.mouse_dw()).max(0).min((ether.reachables.len() - 1) as i32);
							ether.sel_reachable = ether.reachables[i as usize].clone();
						}
					},

					Mode::Descript => {
						// Switch to next description
						if sys.poll_btn(Button::Right) {
							*curr_desc_i += 1;
						}
					},

					_ => ()
				}

				Ok(())
			},

			Self::Unset => Err(format!("{} {}", MID, ERR_NOT_SET_CANNOT_RUN))
		}
    }

}
