/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use core::slice::ChunksMut;

use crate::{
	base::Position,
	sys::System
};

/// Mode of player interaction
#[derive(Clone, Copy, Hash, Eq, PartialEq, Debug)]
pub enum Mode {
	Fly,
	Descript,
	Inventory,
	InventoryCombine,
	TalkListen,
	TalkSpeak,
	Passive
}

/// For Masters to communicate with each other. More "ephemeral" than State.
pub struct Ether {
	pub quit: bool,
	pub meta: bool,
	pub paused: bool, // changes after meta
	pub replay: bool, // when State has been reset (new or loaded), run Verse::play() anew

	pub mode: Mode,

	pub z_buffer: Vec<i64>,
	pub central_look_ray_dist: i64,

	pub reachables: Vec<String>, // entities that allow interaction (look at, use item, ...) and are close enough; to be populated by corresp. masters
    pub sel_reachable: String, // selected reachable

	pub pre_descript_mode: Mode,
	pub descript: String, // id that may be prefixed and suffixed (e.g. add indices for multipart descriptions) by masters to lookup in Lingua

	pub pre_talk_mode: Mode,
	pub talker: String,
	pub speech: String, // what talker says, id that may be prefixed and suffixed
	pub cues: Vec<String>, // what player can say, id-s that may be prefixed
	pub sel_cue: String, // selected cue

	pub music_id: String,
	pub ambient_id: String,
}

impl Ether {
	pub fn new(sys: &System) -> Ether {
		let width = if sys.antialiasing_4x() { sys.width() << 1 } else { sys.width() };
		let height = if sys.antialiasing_4x() { sys.height() << 1 } else { sys.height() };

		Ether {
			quit: false,
			meta: false,
			paused: false,
			replay: false,
			mode: Mode::Passive,
			z_buffer: vec![0; (width as usize) * (height as usize)],
			central_look_ray_dist: std::i64::MAX,
			reachables: Vec::<String>::new(),
			sel_reachable: String::new(),
			pre_descript_mode: Mode::Fly,
			descript: String::new(),
			pre_talk_mode: Mode::Fly,
			talker: String::new(),
			speech: String::new(),
			cues: Vec::<String>::new(),
			sel_cue: String::new(),
			music_id: String::new(),
			ambient_id: String::new()
		}
	}

	pub fn zbuf_raw_chunks_mut(&mut self, pixels_per_chunk: i32) -> ChunksMut<i64> {
		self.z_buffer.chunks_mut(pixels_per_chunk as usize)
	}

	pub fn sync_reach(&mut self, plr_pos: &Position, ent_pos: &Position, reach_dist: i64, id: &String) -> bool {
        if plr_pos.dist_square(ent_pos) <= (reach_dist as i128) * (reach_dist as i128) {
			if self.sel_reachable.is_empty() && self.reachables.is_empty() {
				self.sel_reachable = id.clone();
			}
            self.reachables.push(id.clone());
            true
        } else {
            if self.sel_reachable.eq(id) {
                self.sel_reachable.clear();
            }
            false
        }
    }

	pub fn reachable_ind(&self, reachable: impl ToString) -> Option<usize> {
		let reachable = reachable.to_string();
		if reachable.is_empty() {
			return None
		}
		for i in 0..self.reachables.len() {
			if reachable.eq(&(self.reachables[i])) {
				return Some(i)
			}
		}
		None
	}

	pub fn sel_reachable_ind(&self) -> Option<usize> {
		self.reachable_ind(&(self.sel_reachable))
	}

	pub fn cue_ind(&self, cue: impl ToString) -> Option<usize> {
		let cue = cue.to_string();
		if cue.is_empty() {
			return None
		}
		for i in 0..self.cues.len() {
			if cue.eq(&(self.cues[i])) {
				return Some(i)
			}
		}
		None
	}

	pub fn sel_cue_ind(&self) -> Option<usize> {
		self.cue_ind(&(self.sel_cue))
	}

	pub fn describe(&mut self, descript: impl ToString) {
		self.descript = descript.to_string();
		self.pre_descript_mode = match self.mode {
			Mode::InventoryCombine => Mode::Inventory,
			_ => self.mode
		};
		self.mode = Mode::Descript;
	}

	pub fn talk(&mut self, talker: impl ToString, speech: impl ToString, cues: &[impl ToString]) {
		self.talker = talker.to_string();
		self.speech = speech.to_string();
		self.cues.clear();
		for c in cues {
			self.cues.push(c.to_string());
		}
		if self.cues.len() > 0 {
			self.sel_cue = self.cues[0].clone();
		} else {
			self.sel_cue.clear();
		}

		self.pre_talk_mode = match self.mode {
			Mode::TalkListen | Mode::TalkSpeak => self.pre_talk_mode, // do not change
			_ => self.mode
		};
		self.mode = Mode::TalkListen;
	}

}
