/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

//! Common visualization and input processing

use crate::{
	base::PrependErrorString,
	sys::System,
	cosm::{
		Cosmos,
		Fontset,
		Lingua
	},
	state::State
};

mod ether;
pub use ether::{
	Ether,
	Mode
};

mod antialias;
pub use antialias::MAntialias;
mod hud;
pub use hud::MHUD;
mod inventory;
pub use inventory::MInventory;
mod loop_start;
pub use loop_start::MLoopStart;
mod meta;
pub use self::meta::MMeta;
mod movement;
pub use movement::MMovement;
mod render_balls;
pub use render_balls::MRenderBalls;
mod render_walls;
pub use render_walls::MRenderWalls;
mod sight;
pub use sight::MSight;
mod sync;
pub use sync::MSync;
mod talk;
pub use talk::MTalk;
mod telemetry;
pub use telemetry::MTelemetry;

pub const ERR_ALREADY_SET: &str = "master already set";
pub const ERR_NOT_SET_CANNOT_RUN: &str = "master not set, cannot run";
pub const ERR_NOT_SET_CANNOT_FINISH: &str = "master not set, cannot finish";

pub struct Context<'a> {
	pub sys: &'a mut System,
	pub fonts: &'a Fontset<'a>,
	pub lingua: &'a mut Lingua,
	pub cosm: &'a mut Cosmos,
	pub state: &'a mut State,
	pub ether: &'a mut Ether
}

pub trait Master {
	fn id(&self) -> String {
		String::from("")
	}

	fn start(&mut self, _ctx: &mut Context) -> Result<(), String> {
		Ok(())
	}

	fn run(&mut self, _ctx: &mut Context) -> Result<(), String> {
		Ok(())
	}

	fn finish(&mut self, _ctx: &mut Context) -> Result<(), String> {
		Ok(())
	}

}

pub struct Mastermind {
	masters: Vec<Box<dyn Master>>,
	meta: Box<dyn Master>
}

impl Mastermind {
	pub fn new() -> Mastermind {
		let masters: Vec<Box<dyn Master>> = Vec::new();
		let meta = MMeta::new();
		Mastermind{masters, meta}
	}

	pub fn add(&mut self, mut ms: Vec<Box<dyn Master>>) {
		self.masters.append(&mut ms); // move
	}

	pub fn start(&mut self, ctx: &mut Context) -> Result<(), String> {
		for m in &mut self.masters {
			m.start(ctx).pre_err(format!("cannot start {} master", m.id()))?;
		}
		self.meta.start(ctx).pre_err(format!("cannot start {} master", self.meta.id()))?;
		Ok(())
	}

	pub fn run(&mut self, ctx: &mut Context) -> Result<(), String> {
		match ctx.ether.meta {
			false => {
				for m in &mut self.masters {
					m.run(ctx).pre_err(format!("cannot run {} master", m.id()))?;
				}
			},
			true => self.meta.run(ctx).pre_err(format!("cannot run {} master", self.meta.id()))?
		}
		Ok(())
	}

	pub fn finish(&mut self, ctx: &mut Context) -> Result<(), String> {
		for m in &mut self.masters {
			m.finish(ctx).pre_err(format!("cannot finish {} master", m.id()))?;
		}
		self.meta.finish(ctx).pre_err(format!("cannot finish {} master", self.meta.id()))?;
		Ok(())
	}

	pub fn add_common_pre(&mut self) {
        self.add(vec![
            MLoopStart::new(),

            MSight::new(),

            MMovement::new()
        ]);
    }

    pub fn add_common_post(&mut self) {
        self.add(vec![
            MRenderWalls::new(),
            MRenderBalls::new(),
            MAntialias::new(),

            MTalk::new(),
            MInventory::new(),
            MHUD::new(),

            MTelemetry::new(),

            MSync::new(),
        ]);
    }

}
